<?php


namespace Game\Operation\EventLog;


use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Game\Config\GameConfig;
use Game\Constant\ModelTypeDefine;
use Game\Data\AccountData;
use Game\Model\AccountModel;
use Game\Operation\PlatformManager;
use Library\Common\Singletons;

class EventLog_Manager
{
    use Singletons;

    private EventLogHandler $handler;

    protected bool $HasInit = false;

    //region
    protected array $logDic = array();

    public array $headData = array();

    private string $logPrefix = "te";

    private function setLogPrefix(string $prefix)
    {
        $this->logPrefix = $prefix;
    }

    /**
     * 初始化
     */
    public function init()
    {
        if ($this->HasInit) {
            return true;
        }
        $this->initClientData();
        if (PlatformManager::getInstance()->init($this->logPrefix)) {
            $this->HasInit = true;
        }
    }

    public function setHandler($h)
    {
        $this->handler = $h;
    }

    /**
     * 记录日志
     * @param EventLog $log
     */
    public function report(EventLog $log)
    {
        if ($this->HasInit == false) {
            $this->setLogPrefix($log->GetLogType());
            $this->init();
        }
        if ($log == null || $this->handler == null) {
            return;
        }
        $this->handler->report($log);
    }

    /**
     * 获取日志对象
     * @param string $type
     * @return EventLog|null
     */
    public function getEventLog(string $type): ?EventLog
    {
        if ($this->HasInit == false) {
            $this->setLogPrefix($type);
            $this->init();
        }
        if (array_key_exists($type, $this->logDic)) {
            /**
             * @var  EventLog $log
             */
            $log = $this->logDic[$type];
            $log->CleanUp();
            return $log;
        }
        $eventLogType = new EventLogType();
        $logName = $eventLogType->getEventLogTypeName($type);
        if (is_null($logName)) {
            return null;
        }
        $log = new $logName();
        $this->logDic[$type] = $log;
        return $log;
    }

    public function cliInitAccount()
    {
        $this->initClientData();
        /**
         * @var AccountModel $account
         */
        $account = ModelManager::getInstance()->getModel(ModelTypeDefine::ACCOUNT);
        $this->handler->setAccount($account->getPlayerId(), $this->headData["#distinct_id"]);
    }

    public function UpdateHeadData($logArr)
    {
        foreach ($logArr as $k => $v) {
            $this->headData[$k] = $v;
        }
    }

    public function updateUserAttr(array $attr)
    {
        if ($this->HasInit == false) {
            $this->init();
        }
        if ($this->handler == null) {
            return;
        }
        $this->handler->setUserAttr($attr);
    }

    protected function initClientData()
    {
        /**
         * @var AccountModel $account
         */
        $account = ModelManager::getInstance()->getModel(ModelTypeDefine::ACCOUNT);
        $data = $account->getAccountAttr(AccountData::DB_CLIENT_DATA);
        LogMark::getInstance()->markDebug(
            "[EventLog_Manager] playerId",
            array(
                "playerId" => $account->getPlayerId(),
                "data" => $data
            )
        );
        $this->headData = json_decode($data, true);
        $this->headData["serverId"] = GameConfig::getInstance()->SERVER_ID();
        $this->headData["channelcode"] = GameConfig::getInstance()->CHANNEL_ID();
    }

    public function register(array $attr)
    {
        if ($this->HasInit == false) {
            $this->init();
        }
        if ($this->handler == null) {
            return;
        }
        $this->handler->register($attr);
    }

    //endregion


}