<?php

/**
 * 道具商城
 */

namespace Game\Model\Shop;

use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\Logic\TemplateHelp;
use Game\Constant\ConstTemplate\TemplateGoodsShop;
use Game\Constant\GameErrorCode;
use Game\Constant\TemplateDefine;
use Game\Data\Shop\ShopItemData;
use Game\Logic\CommonConstLogic;
use Game\Protobuf\ShopItem;

class ShopItemModel
{
    use TemplateHelp;
    use CommonConstLogic;

    private int $playerId;
    private array $sendShopItem = array();


    public function setPlayerId($playerId)
    {
        $this->playerId = $playerId;
    }

    //获取道具商城道具
    public function getShopItem(): array
    {
        //查询当前货架批次
        $shopItemData = new ShopItemData($this->playerId);
        $shopItem = $shopItemData->searchDB();
        if (empty($shopItem)) {
            //初始道具商城
            return $this->initShopItem();
        }
        return $shopItem;
    }

    //初始商店道具
    private function initShopItem(): array
    {
        $shopItem = array(
            ShopItemData::RefreshDate => Utils::getServerTimestamp()
        );
        $shopItemConfig = $this->getTable(TemplateDefine::TYPE_SHOP_ITEM);
        $now = Utils::getServerTimestamp();
        foreach ($shopItemConfig as $itemId => $shopItemConfigData) {
            //初始符合时间的道具
            if ($now >= $shopItemConfigData[TemplateGoodsShop::StartTime] &&
                $now <= $shopItemConfigData[TemplateGoodsShop::EndTime]
            ) {
                $shopItem[$itemId] = 0;
            }
        }
        $shopItemData = new ShopItemData($this->playerId);
        if (!$shopItemData->updateDB($shopItem)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                "[ShopItemModel] init shop date error",
                $shopItem
            );
            return array();
        }
        return $shopItem;
    }

    //重置每日限购
    public function resetDailyLimit()
    {
        $shopItemData = new ShopItemData($this->playerId);
        $shopItem = $shopItemData->searchDB();
        if (empty($shopItem)) {
            $this->initShopItem();
            return;
        }
        $now = Utils::getServerTimestamp();
        $resetItem = array();   //重置和新增道具
        $delItemIds = array();
        $shopItemConfig = $this->getTable(TemplateDefine::TYPE_SHOP_ITEM);
        $isCrossWeek = Utils::checkIsCrossWeek($shopItem[ShopItemData::RefreshDate]);
        foreach ($shopItemConfig as $itemId => $itemConfig) {
            //检查是否需要下架
            if ($now >= $itemConfig[TemplateGoodsShop::EndTime]) {
                $delItemIds[] = $itemId;
                continue;
            }
            //是否需要新加
            if (!array_key_exists($itemId, $shopItem) && $now >= $itemConfig[TemplateGoodsShop::StartTime]) {
                $resetItem[$itemId] = 0;
                continue;
            }
            //检查每日限购
            if ($itemConfig[TemplateGoodsShop::ShowType] == TemplateGoodsShop::SHOW_TYPE_DAILY_LIMIT) {
                $resetItem[$itemId] = 0;
                continue;
            }
            //检查每周限购
            if ($isCrossWeek && $itemConfig[TemplateGoodsShop::ShowType] == TemplateGoodsShop::SHOW_TYPE_WEEK_LIMIT) {
                $resetItem[$itemId] = 0;
                continue;
            }
        }
        if ($isCrossWeek) {
            $resetItem[ShopItemData::RefreshDate] = Utils::getServerTimestamp();
        }
        //删除过期和重置
        if (!empty($delItemIds)) {
            $shopItemData->delItemId($delItemIds);
        }
        if (!empty($resetItem)) {
            $shopItemData->updateDB($resetItem);
        }
    }

    //跨赛季重置赛季限购
    public function resetSeasonLimit()
    {
        $shopItemData = new ShopItemData($this->playerId);
        $shopItem = $shopItemData->searchDB();
        if (empty($shopItem)) {
            $this->initShopItem();
            return;
        }
        $resetItem = array();   //重置和新增道具
        $shopItemConfig = $this->getTable(TemplateDefine::TYPE_SHOP_ITEM);
        foreach ($shopItemConfig as $itemId => $itemConfig) {
            //检查每日限购
            if ($itemConfig[TemplateGoodsShop::ShowType] == TemplateGoodsShop::SHOW_TYPE_SEASON_LIMIT) {
                $resetItem[$itemId] = 0;
                continue;
            }
        }
        //重置
        if (!empty($resetItem)) {
            $shopItemData->updateDB($resetItem);
        }
    }

    //获取已购买数量
    public function getHasBuyNum(int $id): int
    {
        $shopItem = $this->getShopItem();
        return $shopItem[$id] ?? 0;
    }

    //更新购买数量
    public function updateBuyNum(array $data): bool
    {
        if (empty($data)) {
            return true;
        }
        return (new ShopItemData($this->playerId))->updateDB($data);
    }

    public function addSendShopItem(array $shopItem)
    {
        foreach ($shopItem as $key => $value) {
            if (is_numeric($key)) {
                $sItem = new ShopItem();
                $sItem->setId($key);
                $sItem->setBuyNum($value);
                $this->sendShopItem[] = $sItem;
            }
        }
    }

    public function getSendShopItem(): array
    {
        return $this->sendShopItem;
    }
}
