<?php

/**
 * 地图挑战赛
 * 每期活动数据单独存储,新活动开始直接初始,上一期活动数据保留
 */

namespace Game\Model\Activity;

use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Game\Constant\ConstTemplate\TemplateChallengeTask;
use Game\Constant\ConstTemplate\TemplateChallengeTime;
use Game\Constant\ConstTemplate\TemplateConst;
use Game\Constant\GameConstantDefine;
use Game\Constant\TemplateDefine;
use Game\Data\Activity\MapChallengeData;
use Game\Logic\CommonConstLogic;
use Game\Logic\MailLogic;


class MapChallengeModel
{
    use CommonConstLogic;
    use MailLogic;

    private int $playerId;
    private ?array $mapChallengeData = null;

    public function setPlayerId($playerId)
    {
        $this->playerId = $playerId;
    }

    public function onLoginExec()
    {
        //检查上次活动奖励是否全部领取
        $this->checkLastActivityReward();
    }

    public function getMapChallengeMapData(int $activityId, int $mapId): array
    {
        if (is_null($this->mapChallengeData)) {
            $this->initMapChallengeData($activityId);
        }
        return (new MapChallengeData($this->playerId, $activityId))->getMapKeyValue($mapId, $this->mapChallengeData);
    }

    public function getMapChallengeData(int $activityId): array
    {
        if (is_null($this->mapChallengeData)) {
            $this->initMapChallengeData($activityId);
        }
        return $this->mapChallengeData;
    }

    public function saveMapChallengeData(int $activityId, array $data): bool
    {
        return (new MapChallengeData($this->playerId, $activityId))->saveDB($data);
    }

    private function initMapChallengeData(int $activityId)
    {
        $mapChallengeData = new MapChallengeData($this->playerId, $activityId);
        $data = $mapChallengeData->getDataFromDB();
        if (empty($data)) {
            $data = $mapChallengeData->initData();
            $mapChallengeData->saveDB($data);
        }
        $this->mapChallengeData = $data;
    }

    //登录检查上期奖励是否全部领取,未领取的发邮件
    //只检查前一期
    private function checkLastActivityReward()
    {
        $result = $this->getCommonConstData();
        $activityId = $result[GameConstantDefine::COMMON_KEY_ACTIVITY_MAP_CHALLENGE] ?? 0;
        if ($activityId <= 1) {
            return;
        }
        $lastActivityId = $activityId - 1;
        //检查上一期是否全部领取
        $mapChallengeData = new MapChallengeData($this->playerId, $lastActivityId);
        $data = $mapChallengeData->getDataFromDB();
        if (empty($data)) {
            //未参加上次活动
            return;
        }
        if ($data[MapChallengeData::ALL_REWARD_GET_STATUS] == MapChallengeData::ALL_REWARD_GET_STATUS_OVER) {
            //已全部领取
            return;
        }
        $config = $this->getTitle(TemplateDefine::TYPE_CHALLENGE_TIME, $lastActivityId);
        if (is_null($config)) {
            LogMark::getInstance()->markDebug(
                "[MapChallengeModel] checkLastActivityReward not found config",
                array(
                    "id" => $lastActivityId
                )
            );
            return;
        }
        $reward = array();
        //检查每个地图可领取的奖励
        foreach ($config[TemplateChallengeTime::MapId] as $mapId) {
            //地图配置
            $mapConfig = $this->getTitle(TemplateDefine::TYPE_CHALLENGE_TASK, $mapId);
            if (is_null($mapConfig)) {
                LogMark::getInstance()->markDebug(
                    "[MapChallengeLogic] not found challenge map config",
                    array(
                        "mapId" => $mapId
                    )
                );
                continue;
            }
            [
                $dbKeyMapBestTime,
                $dbKeyMapCompleteTaskNum,
                $dbKeyMapAllCompleteFlag,
                $dbKeyMapCanGetRewardId,
                $dbKeyMapAlreadyGetRewardId
            ] = MapChallengeData::getMapKey($mapId);
            //检查可领取奖励
            $mapCanGetRewardId = $data[$dbKeyMapCanGetRewardId] ?? 0;         //可领取的最大id
            $mapAlreadyGetRewardId = $data[$dbKeyMapAlreadyGetRewardId] ?? 0; //已领取的最大id
            if ($mapCanGetRewardId == $mapAlreadyGetRewardId) {
                //没有可领取的
                continue;
            }
            foreach ($mapConfig as $id => $configData) {
                if ($id > $mapCanGetRewardId) {
                    break;
                }
                if ($mapAlreadyGetRewardId > $id) {
                    continue;
                }
                foreach ($configData[TemplateChallengeTask::Reward] as $rewardId => $num) {
                    isset($reward[$rewardId]) ? $reward[$rewardId] += $num : $reward[$rewardId] = $num;
                }
            }
        }
        //检查大奖是否领取
        if ($data[MapChallengeData::REWARD_STATUS] == MapChallengeData::REWARD_STATUS_CAN_GET) {
            foreach ($config[TemplateChallengeTime::Reward] as $rewardId => $num) {
                isset($reward[$rewardId]) ? $reward[$rewardId] += $num : $reward[$rewardId] = $num;
            }
        }

        //记录领取状态
        $updateData = array(
            MapChallengeData::ALL_REWARD_GET_STATUS => MapChallengeData::ALL_REWARD_GET_STATUS_OVER,
            MapChallengeData::ALL_REWARD_GET_TIME => Utils::getServerTimestamp(),    //领取时间
        );
        $mapChallengeData->saveDB($updateData);
        //检查发送邮件
        if (!empty($reward)) {
            $mailTplId = $this->getTerm(TemplateDefine::TYPE_CONST,
                                        TemplateConst::Const_Map_Challenge_Mail_id,
                                        TemplateConst::ConstNum);
            $this->createPlayerMail($mailTplId, Utils::getServerTimestamp(), $reward);
        }
    }
}
