<?php

/**
 * 任务
 */

namespace Game\Logic\Task;

use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Framework\Logic\TemplateHelp;
use Game\Constant\ClientErrorCode;
use Game\Constant\ConstTemplate\TemplateTask;
use Game\Constant\EventTypeDefine;
use Game\Constant\GameConstantDefine;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Data\TaskData;
use Game\Logic\ActiveLogic;
use Game\Logic\CarLogic;
use Game\Logic\EventLogic;
use Game\Logic\ItemLogic;
use Game\Method\Player\AddPlayerExp;
use Game\Model\Car\CarModel;
use Game\Model\Task\ActiveModel;
use Game\Model\Task\Manager\BranchTaskManager;
use Game\Model\Task\Manager\DailyChallengeTaskManager;
use Game\Model\Task\Manager\DailyTaskManager;
use Game\Model\Task\Manager\FriendIntimacyTaskManager;
use Game\Model\Task\Manager\MainTaskManager;
use Game\Model\Task\Manager\SeasonTaskManager;
use Game\Model\Task\Manager\TaskMangerModelBase;
use Game\Model\Task\Manager\WeekTaskManager;
use Game\Model\Task\WarTokenModel;

trait TaskLogic
{
    use TemplateHelp;
    use ItemLogic;
    use EventLogic;
    use CarLogic;
    use AddPlayerExp;
    use ActiveLogic;

    public function getDailyTask(): array
    {
        /**
         * @var DailyTaskManager $dailyTaskManager
         */
        $dailyTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_DAILY);
        return $dailyTaskManager->getDailyTask();
    }

    public function getDailyChallengeTask(): array
    {
        /**
         * @var DailyChallengeTaskManager $dayChallengeTaskManager
         */
        $dayChallengeTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_DAILY_CHALLENGE);
        return $dayChallengeTaskManager->getDailyChallengeTask();
    }

    public function getWelfareWeekTask(): array
    {
        /**
         * @var WeekTaskManager $WeekTaskManager
         */
        $WeekTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_WELFARE_WEEK);
        return $WeekTaskManager->getWelfareWeekTask();
    }

    public function getMainTask(): array
    {
        /**
         * @var MainTaskManager $mainTaskManager
         */
        $mainTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_MAIN);
        return $mainTaskManager->getMainTask();
    }

    //获取任务--确认开启后调用
    public function getFriendIntimacyRewardTask(): array
    {

        /**
         * @var FriendIntimacyTaskManager $friendIntimacyTaskManager
         */
        $friendIntimacyTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_FRIEND_INTIMACY_REWARD);
        return $friendIntimacyTaskManager->getFriendIntimacyTask();
    }

    //获取支线任务--检查车是否解锁
    private function checkHasCar(int $carId): bool
    {
        /**
         * @var CarModel $carModel
         */
        $carModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CAR);
        $car = $carModel->searchCarByCarID($carId);
        if (is_null($car)) {
            return false;
        }
        return true;
    }

    public function getBranchTask(int $carId): array
    {
        /**
         * @var BranchTaskManager $branchTaskManager
         */
        $branchTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_BRANCH);
        return $branchTaskManager->getBranchTaskByCarId($carId);
    }

    //未开启的周 返回空
    public function getSeasonTask(int $week): array
    {
        /**
         * @var SeasonTaskManager $seasonTaskManager
         */
        $seasonTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_SEASON);
        return $seasonTaskManager->getSeasonTaskByWeek($week);
    }

    public function checkTaskExist(int $taskId): int
    {
        $taskConfig = $this->getTitle(TemplateDefine::TYPE_TASK, $taskId);
        if (is_null($taskConfig)) {
            LogMark::getInstance()->markError(
                GameErrorCode::TASK_ID_NOT_FOUND,
                "[TaskLogic] taskId not found",
                array(
                    "taskId" => $taskId
                )
            );
            return ClientErrorCode::TASK_NOT_FOUND;
        }
        return ClientErrorCode::CLIENT_SUCCESS;
    }

    //战令任务 检查是否开启高级
    public function checkHasOpenExpert(): bool
    {
        /**
         * @var WarTokenModel $warTokenModel
         */
        $warTokenModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WAR_TOKEN);
        return $warTokenModel->getWarTokenData()->expertFlag;
    }

    //战令任务 检查对应周是否开启
    public function checkHasOpenWeek(int $week): bool
    {
        /**
         * @var SeasonTaskManager $seasonTaskManager
         */
        $seasonTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_SEASON);
        $openWeek = $seasonTaskManager->getOpenSeasonWeek();
        return $week <= $openWeek;
    }

    //查询任务
    public function searchTask(int $taskId): ?object
    {
        /**
         * @var TaskMangerModelBase $taskBaseModel
         */
        $taskBaseModel = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_BASE);
        return $taskBaseModel->getTask($taskId);
    }

    //校验完成后,提交任务
    public function submitTask(TaskData $taskData)
    {
        /**
         * @var TaskMangerModelBase $taskBaseModel
         */
        $taskBaseModel = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_BASE);
        $taskBaseModel->completeTask($taskData);
        //触发完成任务数量
        $param = array(
            GameConstantDefine::EVENT_KEY_COMP_TASK_TYPE => array(
                $taskData->taskConfig[TemplateTask::TaskType] => 1
            )
        );
        $this->triggerEvent(EventTypeDefine::EVENT_TYPE_TASK_COMP_NUM, $param);
        //加道具
        [$itemReward, $numReward, $exp] = $taskBaseModel->getCompleteReward();
        $this->setLogAddItemSource(GameConstantDefine::ITEM_ADD_SOURCE_TASK);
        $this->gainItemArr($itemReward);
        //完成任务 后续操作
        switch ($taskData->taskConfig[TemplateTask::TaskType]) {
            case TemplateTask::TASK_TYPE_MAIN:
                $this->completeMainTask($taskData, $taskBaseModel);
                //加玩家经验
                $this->addPlayerExp($numReward);
                break;
            case TemplateTask::TASK_TYPE_BRANCH:
                $this->completeBranchTask($taskData, $taskBaseModel);
                //赛车支线任务 加熟练度
                if ($numReward > 0) {
                    $this->addCarProficiency($taskData->carId, $numReward);
                }
                break;
            case TemplateTask::TASK_TYPE_SEASON_WEEK:
            case TemplateTask::TASK_TYPE_SEASON:
                $this->completeSeasonTask($numReward);
                break;
            case TemplateTask::TASK_TYPE_DAILY_CHALLENGE:
                //加活跃度
                if ($numReward > 0) {
                    $this->addActivePoint($numReward);
                    /**
                     * @var ActiveModel $activeModel
                     */
                    $activeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_ACTIVE);
                    $activeModel->sendActiveMessage();
                }
                //加玩家经验
                $this->addPlayerExp($exp);
                break;
            case TemplateTask::TASK_TYPE_FRIEND_INTIMACY:
                $this->addFriendIntimacyRewardScore($numReward);
                break;
        }
        //发消息
        $taskBaseModel->sendChangeTaskMessage();
    }

    //主线任务--完成后删除
    private function completeMainTask(TaskData $taskData, TaskMangerModelBase $taskBaseModel)
    {
        $taskBaseModel->delTask($taskData->taskId);
    }



    //支线任务--完成后检查
    private function completeBranchTask(TaskData $taskData, TaskMangerModelBase $taskBaseModel)
    {
        //检查任务类型
        if ($taskData->taskConfig[TemplateTask::ConditionType] == TemplateTask::TASK_BRANCH_MAIN_CONDITION_TYPE) {
            //主任务--接下一级任务
            /**
             * @var BranchTaskManager $branchTaskManager
             */
            $branchTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_BRANCH);
            $newTaskList = $branchTaskManager->completeBranchMainTask($taskData->taskConfig[TemplateTask::GroupType], $taskData->taskConfig[TemplateTask::GroupId]);
            foreach ($newTaskList as $newTaskData) {
                $taskBaseModel->addChangeTask($newTaskData);
            }
        } else {
            //子任务--触发更新主任务进度
            $data = array(
                GameConstantDefine::EVENT_KEY_TASK_ID => $taskData->taskId
            );
            $this->triggerEvent(EventTypeDefine::EVENT_TYPE_TASK_BRANCH_MAIN, $data);
        }
    }

    //战令任务--完成后更新经验
    private function completeSeasonTask(int $addExp)
    {
        if ($addExp <= 0) {
            return;
        }
        /**
         * @var WarTokenModel $warTokenModel
         */
        $warTokenModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WAR_TOKEN);
        $warTokenModel->addWarTokenExp($addExp);
        $warTokenModel->sendMessage();
    }

    //GM修改成长(主线)任务
    public function gmCompleteMainTask(int $lv)
    {
        /**
         * @var MainTaskManager $mainTaskManager
         */
        $mainTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_MAIN);
        $mainTaskManager->gmCompleteMainTask($lv);
    }

    //GM完成任务--false为任务不存在
    public function gmCompleteTask(int $taskId): bool
    {
        /**
         * @var TaskMangerModelBase $taskBaseModel
         */
        $taskBaseModel = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_BASE);
        $taskData = $taskBaseModel->getTask($taskId);
        if (is_null($taskData)) {
            return false;
        }
        if ($taskData->status == TaskData::STATUS_DOING) {
            $taskData->status = TaskData::STATUS_DONE;
            $taskData->updateTask();
        }
        return true;
    }
}
