<?php

/**
 * 任务逻辑处理
 */

namespace Game\Logic\Task;


use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Game\Constant\ConstTemplate\TemplateTask;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TaskLogicDefine;
use Game\Data\TaskData;
use Game\Logic\AccountLogic;
use Game\Model\EventModel;
use Game\Model\Task\Logic\TaskLogicBase;
use Game\Model\Task\Logic\TaskLogicManager;
use Game\Data\AccountData;

trait TaskEventLogic
{
    use AccountLogic;

    //初始任务
    public function initTaskLogic(TaskData $taskData, array $param, bool $isRegister = false): bool
    {
        //根据任务条件,查找对应处理逻辑
        $eventId = $this->getEventId($taskData);
        if ($eventId == 0) {
            return false;
        }
        $taskLogic = $this->getTaskLogic($eventId);
        if (is_null($taskLogic)) {
            return false;
        }
        /**
         * @var TaskLogicBase $taskLogic
         */
        if ($isRegister) {
            $taskLogic->setIsRegister();
        }
        $taskLogic->getDefaultValue($taskData, $param);
        if ($taskData->status == TaskData::STATUS_DOING) {
            //注册事件
            $data = array(
                $taskData->taskId => 0
            );
            return $this->registerEvent($eventId, $data, $isRegister);
        }
        return true;
    }

    // 第一次登录检测周任务事件逻辑
    public function changeWeekTaskLogic(TaskData $taskData, array $param): bool
    {
        //根据任务条件,查找对应处理逻辑
        $eventId = $this->getEventId($taskData);
        if ($eventId == 0) {
            return false;
        }
        $taskLogic = $this->getTaskLogic($eventId);
        if (is_null($taskLogic)) {
            return false;
        }
        /**
         * @var TaskLogicBase $taskLogic
         */
        $taskLogic->getDefaultValue($taskData, $param);
        //注册事件
        $data = array(
            $taskData->taskId => 0
        );
        $day = date("w");
        $cfg = $this->getTaskConfig($taskData->taskId);
        // 不需要排除的event
        $execWeekTask = [6, 0];
        if(in_array($cfg[TemplateTask::GroupId], $execWeekTask)) {
            return true;
        }
        if($cfg[TemplateTask::GroupId] == $day) {
            return $this->registerEvent($eventId, $data);
        }
        $data = [
            (string)$this->getModelType() . "-" . (string)$taskData->taskId
        ];
        $this->removeEvent($eventId, $data);
        return true;
    }

    // 周任务添加两个累加不删除的event
    public function changeWeekLoginTaskLogic(TaskData $taskData, array $param, bool $isRegister = false): bool
    {
        //根据任务条件,查找对应处理逻辑
        $eventId = $this->getEventId($taskData);
        if ($eventId == 0) {
            return false;
        }
        $taskLogic = $this->getTaskLogic($eventId);
        if (is_null($taskLogic)) {
            return false;
        }
        /**
         * @var TaskLogicBase $taskLogic
         */
        $taskLogic->getDefaultValue($taskData, $param);
        //注册事件
        $data = array(
            $taskData->taskId => 0
        );
        $cfg = $this->getTaskConfig($taskData->taskId);
        // 不需要排除的event
        $execWeekTask = [6, 0];
        $day = (int)date("w");
        if(in_array($cfg[TemplateTask::GroupId], $execWeekTask) || $day === $cfg[TemplateTask::GroupId]) {
            return $this->registerEvent($eventId, $data, $isRegister);
        }
        return true;
    }

    //触发任务
    public function triggerTask(TaskData $taskData, array $param)
    {
        //根据任务条件,查找对应处理逻辑
        $eventId = $this->getEventId($taskData);
        if ($eventId == 0) {
            return ;
        }
        $taskLogic = $this->getTaskLogic($eventId);
        if (is_null($taskLogic)) {
            return ;
        }
        /**
         * @var TaskLogicBase $taskLogic
         */
        $taskLogic->updateTask($taskData, $param);
        if ($taskData->status == TaskData::STATUS_DONE) {
            //移除事件
            $data = array(
                (string)$this->getModelType() . "-" . (string)$taskData->taskId
            );
            $this->removeEvent($eventId, $data);
        }
    }

    //根据任务条件 查询EventId
    private function getEventId(TaskData $taskData): int
    {
        $tld = new TaskLogicDefine();
        $eventId = $tld->getEventId($taskData->taskConfig[TemplateTask::ConditionType]);
        if ($eventId == 0) {
            LogMark::getInstance()->markError(
                GameErrorCode::TASK_NOT_FOUND_EVENT,
                "[TaskEventLogic] not found Event",
                array(
                    "ConditionType" => $taskData->taskConfig[TemplateTask::ConditionType]
                )
            );
        }
        return $eventId;
    }

    private function getTaskLogic(int $eventId): ?object
    {
        return TaskLogicManager::getInstance()->getTaskLogic($eventId, $this->playerId);
    }

    //注册事件
    private function registerEvent(int $eventId, array $data, bool $isRegister = false): bool
    {
        /**
         * @var EventModel $eventModel
         */
        $eventModel = ModelManager::getInstance()->getModel(ModelTypeDefine::EVENT);
        if ($isRegister) {
            return $eventModel->pipeAddRegisterEvent($eventId, $this->getModelType(), $data);
        }
        return $eventModel->registerEventData($eventId, $this->getModelType(), $data);
    }

    //批量注册事件
    protected function pipeRegisterEventData()
    {
        /**
         * @var EventModel $eventModel
         */
        $eventModel = ModelManager::getInstance()->getModel(ModelTypeDefine::EVENT);
        $eventModel->pipeRegisterEvent();
    }

    //根据任务id移除事件
    public function removeEventByTaskId(TaskData $taskData): bool
    {
        $eventId = $this->getEventId($taskData);
        if ($eventId == 0) {
            return false;
        }
        $taskLogic = $this->getTaskLogic($eventId);
        if (is_null($taskLogic)) {
            return false;
        }
        /**
         * @var TaskLogicBase $taskLogic
         */
        $data = array(
            (string)$this->getModelType() . "-" . (string)$taskData->taskId
        );
        return $this->removeEvent($eventId, $data);
    }

    private function getModelType(): int
    {
        return ModelTypeDefine::TASK_BASE;
    }

    //查询默认驾驶车Id
    private function getDefaultCarId(): int
    {
        $carId = $this->getAccountProperty($this->playerId, AccountData::DB_DEFAULT_CAT_TPL);
        if (is_null($carId)) {
            return 0;
        }
        return $carId;
    }
}
