<?php

/**
 * 赛季
 */

namespace Game\Logic;

use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Framework\Logic\CommonDataLogic;
use Framework\Logic\TemplateHelp;
use Game\Constant\ConstTemplate\TemplateConst;
use Game\Constant\GameConstantDefine;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Data\WarTokenData;
use Game\Model\MoneyModel;
use Game\Model\PaiweiLevelModel;
use Game\Model\Season\SeasonStatisticsModel;
use Game\Model\Shop\ShopItemModel;
use Game\Model\Shop\ShopSeasonModel;
use Game\Model\Shop\ShopZLModel;
use Game\Model\Task\Manager\SeasonTaskManager;
use Game\Model\Task\WarTokenModel;

trait SeasonLogic
{
    use TemplateHelp;
    use MailLogic;
    use CommonDataLogic;

    //获取当前赛季Id
    public function getSeasonId(): int
    {
        $seasonId = $this->getCommonValueByModel(ModelTypeDefine::SEASON);
        if (is_null($seasonId)) {
            LogMark::getInstance()->markError(
                GameErrorCode::SEASON_ID_NOT_FOUND,
                "[SeasonLogic] SeasonId not found"
            );
            $seasonId = 0;
        }
        return $seasonId;
    }

    //获取赛季状态
    public function getSeasonStatus(): int
    {
        $seasonStatus = $this->getCommonValueByModel(ModelTypeDefine::SEASON_STATUS);
        if (is_null($seasonStatus)) {
            LogMark::getInstance()->markError(
                GameErrorCode::SEASON_ID_NOT_FOUND,
                "[SeasonLogic] Season Status error"
            );
            $seasonStatus = GameConstantDefine::SEASON_STATUS_CLOSE;
        }
        return $seasonStatus;
    }

    //重置赛季相关信息
    public function resetSeasonData(int $seasonId)
    {
        LogMark::getInstance()->markInfo(
            "Start reset Season Data,new SeasonId:" . strval($seasonId)
        );
        //检查重置排位信息
        $this->resetPaiweiData($seasonId);
        //重置赛季统计数据
        $this->resetSeasonStatisticsData();
        //重置赛季任务 战令
        $this->resetSeasonTaskData();
        //重置赛季商店
        $this->resetSeasonShop($seasonId);
        //重置战令商店
        $this->resetZLShop($seasonId);
        //重置道具商店赛季限购
        $this->resetShopItemSeasonLimit();
    }

    //重置排位信息
    private function resetPaiweiData(int $seasonId)
    {
        /**
         * @var PaiweiLevelModel $paiweiModel
         */
        $paiweiModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PAIWEI_LEVEL);
        $paiweiData = $paiweiModel->searchLevelData();
        if ($paiweiData->seasonId == $seasonId) {
            return;
        }
        $paiweiModel->resetPaiweiLevel($seasonId);
    }

    //重置赛季统计信息
    private function resetSeasonStatisticsData()
    {
        /**
         * @var SeasonStatisticsModel $ssModel
         */
        $ssModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SEASON_STATISTICS);
        $ssModel->resetNowSeasonData();
    }

    //重置赛季任务 战令
    private function resetSeasonTaskData()
    {
        //完成未领取的战令任务,领奖发邮件
        $this->checkDoneSeasonTask();
        LogMark::getInstance()->markInfo("reset season Task");
        //重置战令信息
        /**
         * @var WarTokenModel $warTokenModel
         */
        $warTokenModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WAR_TOKEN);
        $warTokenModel->reset();

        // 重置用户战令币
        /**
         * @var MoneyModel $moneyModel
         */
        $moneyModel = ModelManager::getInstance()->getModel(ModelTypeDefine::MONEY);
        $moneyModel->reset(MoneyModel::TYPE_ZL);

        //重置战令任务
        /**
         * @var SeasonTaskManager $seasonTaskManager
         */
        $seasonTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_SEASON);
        $seasonTaskManager->resetSeasonTask();
        //重置赛季周索引
        $warTokenModel->resetSeasonWeek();
        // 赛季 战令币重置
    }

    //重置赛季商店
    private function resetSeasonShop(int $seasonId)
    {
        /**
         * @var ShopSeasonModel $shopSeasonModel
         */
        $shopSeasonModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SHOP_SEASON);
        $shopSeasonModel->resetShopSeasonData($seasonId);
    }

    //重置赛季商店
    private function resetZLShop(int $seasonId)
    {
        /**
         * @var ShopZLModel $zlModel
         */
        $zlModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SHOP_ZL);
        $zlModel->resetShopZLData($seasonId);
    }

    //重置道具商店赛季限购
    private function resetShopItemSeasonLimit()
    {
        /**
         * @var ShopItemModel $shopItemModel
         */
        $shopItemModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SHOP_ITEM);
        $shopItemModel->resetSeasonLimit();
    }

    //检查全部完成未领取的战令任务--发邮件
    private function checkDoneSeasonTask()
    {
        /**
         * @var SeasonTaskManager $seasonTaskManager
         */
        $seasonTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_SEASON);
        [$itemReward, $numReward] = $seasonTaskManager->getDoneNotGetTaskReward();
        //检查战令奖励
        /**
         * @var WarTokenModel $warTokenModel
         */
        $warTokenModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WAR_TOKEN);
        //最大可领取id
        $canGatLv = $warTokenModel->getWarTokenCanLv($numReward);
        //检查是否可领取
        $warTokenData = $warTokenModel->getWarTokenData();
        $warTokenReward = array();
        $primaryIdList = $warTokenModel->getPrimaryIdList();
        if (count($primaryIdList) < $canGatLv) {
            //领取初级奖励
            for ($i = 1; $i <= $canGatLv; $i++) {
                if (in_array($i, $primaryIdList)) {
                    continue;
                }
                $warTokenModel->getPrimaryReward($i, $warTokenReward);
            }
        }
        //领取高级奖励
        if ($warTokenData->expertFlag) {
            $expertIdList = $warTokenModel->getExpertIdList();
            if (count($expertIdList) < $canGatLv) {
                for ($i = 1; $i <= $canGatLv; $i++) {
                    if (in_array($i, $expertIdList)) {
                        continue;
                    }
                    $warTokenModel->getExpertReward($i, $warTokenReward, $warTokenData->expertFlag);
                }
            }
        }
        //整理道具
        $reward = $this->tidyWarTokenReward($itemReward, $warTokenReward);
        if (!empty($reward)) {
            LogMark::getInstance()->markInfo("Season Task,get Done but not get reward");
            //发邮件
            $this->sendUnReceiveMail($reward);
        }
    }

    private function tidyWarTokenReward(array $reward1, array $reward2): array
    {
        $reward = array();
        foreach ($reward1 as $itemId => $num) {
            isset($reward[$itemId]) ?
                $reward[$itemId] += $num :
                $reward[$itemId] = $num;
        }
        foreach ($reward2 as $itemId => $num) {
            isset($reward[$itemId]) ?
                $reward[$itemId] += $num :
                $reward[$itemId] = $num;
        }
        return $reward;
    }

    //发送未领奖邮件
    private function sendUnReceiveMail(array $reward)
    {
        $mailTplId = $this->getTerm(
            TemplateDefine::TYPE_CONST,
            TemplateConst::Const_Task_Season_Reward_Mail_id,
            TemplateConst::ConstNum
        );
        $this->createPlayerMail($mailTplId, Utils::getServerTimestamp(), $reward);
    }

    // 赛季分更新
    public function updateSeasonScore(int $type, int $playerID = 0) {
        /**
         * @var PaiweiLevelModel $paiweiModel
         */
        $paiweiModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PAIWEI_LEVEL);
        $paiweiModel->updateSeasonScore($type, $playerID);
    }
}
