<?php
/**
 * 房间内信息修改类
 */
namespace Game\Logic\RoomCustomize;

use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Framework\Network\SendMessage;
use Framework\Logic\PacketCacheLogic;
use Framework\Logic\TemplateHelp;
use Game\Constant\ClientErrorCode;
use Game\Constant\ConstTemplate\TemplateCompetitionMode;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Data\AccountData;
use Game\Data\RoomCustomizeData;
use Game\Data\RoomData;
use Game\Logic\CarLogic;
use Game\Logic\GenerateAiLogic;
use Game\Model\AccountModel;
use Game\Model\RoomCustomizeModel;
use Game\Logic\AccountLogic;
use Game\Model\RoomModel;
use Game\Protobuf\GCSaveMapID;
use Game\Protobuf\PacketId;
use Gm\Config\ErrorCode;


trait RoomInfoLogic {
    use TemplateHelp;
    use CarLogic;
    use PacketCacheLogic;
    use GenerateAiLogic;
    use AccountLogic;

    public int $playerID;
    // 房主更换地图
    public function updateRoomMap(int $roomID, int $mapID): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change map search customize room error!', (array)$room
            );
            $roomModel->GCUpdateRoomMap(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        $playerList = $room->playerInfo;
        if(!$this->isOwner($playerList)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[roomInfoLogic] not permission change map!', (array)$room
            );
            $roomModel->GCUpdateRoomMap(ClientErrorCode::ERROR_NOT_PERMISSION);
            return false;
        }

        if($room->mapID == $mapID) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] not repeat select map!', (array)$room
            );
            $roomModel->GCUpdateRoomMap(ClientErrorCode::ERROR_ROOM_NOT_REPEAT_CHANGE);
            return false;
        }

        $mapCfg = $this->getTitle(TemplateDefine::TYPE_PVP_MAP, $mapID);
        if(is_null($mapCfg)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] map not exist!', ['mapID' => $mapID]
            );
            $roomModel->GCUpdateRoomMap(ClientErrorCode::ERROR_ROOM_MAP_NOT_EXIST);
            return false;
        }

        if(!$roomModel->saveRoomMap($room, $mapID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update customize room map error!', (array)$room
            );
            $roomModel->GCUpdateRoomMap(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

//        $roomModel->GCUpdateRoomMap(ClientErrorCode::CLIENT_SUCCESS);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCAddUpdateRoomMap(ClientErrorCode::CLIENT_SUCCESS, $mapID, $id);
        }
        return true;
    }


    // 房主更换模式
    public function updateRoomMode(int $roomID, int $mode): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $modeCfg = $this->getTitle(TemplateDefine::TYPE_COMPETITION_MODE, $mode);
        if(is_null($modeCfg)) {
            logMark::getInstance()->markError(
                GameErrorCode::SEARCH_CONFIG_INFO_FAILED,
                '[RoomCustomize-RoomInfoLogic] update customize room mode search mode cfg error!',
                ['modeType' => $mode]
            );
            $roomModel->GCUpdateRoomMode(ClientErrorCode::ERROR_ROOM_MODE_NOT_EXIST);
            return false;
        }

        if(!$this->modeTypeIsOpen($modeCfg)) {
            logMark::getInstance()->markError(
                GameErrorCode::ROOM_MODE_TYPE_NOT_OPEN,
                '[RoomCustomize-RoomInfoLogic] modeType not open error!',
                ['modeType' => $mode]
            );
            $roomModel->GCUpdateRoomMode(ClientErrorCode::ERROR_ROOM_MODE_TYPE_NOT_OPEN);
            return false;
        }

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change mode search customize room error!', (array)$room
            );
            $roomModel->GCUpdateRoomMode(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        $playerList = $room->playerInfo;
        if(!$this->isOwner($playerList)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[roomInfoLogic] not permission change mode!', (array)$room
            );
            $roomModel->GCUpdateRoomMap(ClientErrorCode::ERROR_NOT_PERMISSION);
            return false;
        }

        if($room->modeType == $mode) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[customize logic] customize not repeat select mode!', (array)$room
            );
            $roomModel->GCUpdateRoomMode(ClientErrorCode::ERROR_ROOM_NOT_REPEAT_CHANGE);
            return false;
        }

        if(!$roomModel->saveRoomMode($room, $mode)) {
            logMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] owner change mode error!',
                ['roomID' => $roomID, 'mode' => $mode]
            );
            $roomModel->GCUpdateRoomMode(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

//        $roomModel->GCUpdateRoomMode(ClientErrorCode::CLIENT_SUCCESS, $mode);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCAddUpdateRoomMode(ClientErrorCode::CLIENT_SUCCESS, $mode, $id);
        }
        return true;
    }

    // 判断模式是否开启
    public function modeTypeIsOpen(array $modeCfg): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        return $modeCfg[TemplateCompetitionMode::OpenIf] == $roomModel::MODE_OPEN;
    }

    // 房主更换房间名称
    public function updateRoomName(int $roomID, string $name): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change name search customize room error!', (array)$room
            );
            $roomModel->GCUpdateRoomName(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        $playerList = $room->playerInfo;
        if(!$this->isOwner($playerList)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[roomInfoLogic] not permission change customize room name!', (array)$room
            );
            $roomModel->GCUpdateRoomName(ClientErrorCode::ERROR_NOT_PERMISSION);
            return false;
        }

        if($room->roomName == $name) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] not repeat select name!', (array)$room
            );
            $roomModel->GCUpdateRoomName(ClientErrorCode::ERROR_ROOM_NOT_REPEAT_CHANGE);
            return false;
        }

        if(!$roomModel->saveRoomName($room, $name)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update customize room name error!', (array)$room
            );
            $roomModel->GCUpdateRoomName(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

//        $roomModel->GCUpdateRoomName(ClientErrorCode::CLIENT_SUCCESS);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCAddUpdateRoomName(ClientErrorCode::CLIENT_SUCCESS, $name, $id);
        }
        return true;
    }

    // 房主ai设置
    public function updateRoomAI(int $roomID, bool $ai): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change ai search customize room error!', (array)$room
            );
            $roomModel->GCUpdateRoomAI(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        $playerList = $room->playerInfo;
        if(!$this->isOwner($playerList)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[roomInfoLogic] not permission change customize room ai!', (array)$room
            );
            $roomModel->GCUpdateRoomAI(ClientErrorCode::ERROR_NOT_PERMISSION);
            return false;
        }

        if($room->ai == $ai) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] not repeat select ai!', (array)$room
            );
            $roomModel->GCUpdateRoomAI(ClientErrorCode::ERROR_ROOM_NOT_REPEAT_CHANGE);
            return false;
        }

        if(!$roomModel->saveRoomAI($room, $ai)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update customize room ai error!', (array)$room
            );
            $roomModel->GCUpdateRoomAI(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

//        $roomModel->GCUpdateRoomAI(ClientErrorCode::CLIENT_SUCCESS);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCAddUpdateRoomAI(ClientErrorCode::CLIENT_SUCCESS, $ai, $id);
        }
        return true;
    }

    // 房主密码设置
    public function updateRoomPwd(int $roomID, string $password): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change password search customize room error!', (array)$room
            );
            $roomModel->GCUpdateRoomPassword(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        $playerList = $room->playerInfo;
        if(!$this->isOwner($playerList)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[roomInfoLogic] not permission change customize room pwd!', (array)$room
            );
            $roomModel->GCUpdateRoomPassword(ClientErrorCode::ERROR_NOT_PERMISSION);
            return false;
        }

        if($room->password == $password) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] not repeat select password!', (array)$room
            );
            $roomModel->GCUpdateRoomPassword(ClientErrorCode::ERROR_ROOM_NOT_REPEAT_CHANGE);
            return false;
        }

        if(!$roomModel->saveRoomPassword($room, $password)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update customize room pwd error!', (array)$room
            );
            $roomModel->GCUpdateRoomPassword(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

//        $roomModel->GCUpdateRoomPassword(ClientErrorCode::CLIENT_SUCCESS);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCAddUpdateRoomPassword(ClientErrorCode::CLIENT_SUCCESS, $password, $id);
        }
        return true;
    }

    // 房主更换房间状态
    public function updateRoomState(int $roomID, int $state): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change customize room state search room error!', (array)$room
            );
            $roomModel->GCUpdateRoomState(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        $playerList = $room->playerInfo;
        if(!$this->isOwner($playerList)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[roomInfoLogic] not permission change customize room state!', (array)$room
            );
            $roomModel->GCUpdateRoomMap(ClientErrorCode::ERROR_NOT_PERMISSION);
            return false;
        }

        if($room->roomState == $state) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] not repeat select name!', (array)$room
            );
            $roomModel->GCUpdateRoomState(ClientErrorCode::ERROR_ROOM_NOT_REPEAT_CHANGE);
            return false;
        }

        if(!$roomModel->saveRoomState($room, $state)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update customize room state error!', (array)$room
            );
            $roomModel->GCUpdateRoomState(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

//        $roomModel->GCUpdateRoomState(ClientErrorCode::CLIENT_SUCCESS);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCAddUpdateRoomState(ClientErrorCode::CLIENT_SUCCESS, $state, $id);
        }
        return true;
    }

    // 房主密码可见修改
    public function updateRoomIsShowPwd(int $roomID, bool $state): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change customize room is show pwd search room error!', (array)$room
            );
            $roomModel->GCUpdateRoomIsShowPwd(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        $playerList = $room->playerInfo;
        if(!$this->isOwner($playerList)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[roomInfoLogic] not permission change customize room is show pwd!', (array)$room
            );
            $roomModel->GCUpdateRoomIsShowPwd(ClientErrorCode::ERROR_NOT_PERMISSION);
            return false;
        }

        if($room->isShowPwd == $state) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] not repeat select name!', (array)$room
            );
            $roomModel->GCUpdateRoomIsShowPwd(ClientErrorCode::ERROR_ROOM_NOT_REPEAT_CHANGE);
            return false;
        }

        if(!$roomModel->saveRoomPwdIsShow($room, $state)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update customize room state error!', (array)$room
            );
            $roomModel->GCUpdateRoomIsShowPwd(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

//        $roomModel->GCUpdateRoomState(ClientErrorCode::CLIENT_SUCCESS);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCAddUpdateRoomIsShowPwd(ClientErrorCode::CLIENT_SUCCESS, $state, $id);
        }
        return true;
    }

    // 判断是否是房主
    public function isOwner(array $playerList): bool {
        array_multisort(array_column($playerList, RoomData::JOIN_TIME), SORT_ASC, $playerList);
        return $playerList[0][RoomData::PLAYER_ID] == $this->playerID;
    }

    // 玩家换车
    public function updatePlayerCar(int $roomID, $carID, $carUID): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change player car search customize room error!', (array)$room
            );
            $roomModel->GCChangePlayerCar(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        if(!$roomModel->savePlayerCar($room, $carID, $carUID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update player car error!', (array)$room
            );
            $roomModel->GCChangePlayerCar(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }
        $code = ClientErrorCode::CLIENT_SUCCESS;
        if(!$roomModel->replacePlayerCar($carUID, $code)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] player car replace error!',
                ['tplID' => $carID, 'carUID' => $carUID]
            );
            $roomModel->GCChangePlayerCar($code);
            return false;
        }

//        $roomModel->GCChangePlayerCar(ClientErrorCode::CLIENT_SUCCESS);
        foreach($room->playerInfo as $id => $p) {
//            if($p[RoomData::PLAYER_ID] == $this->playerID) {
//                continue;
//            }
            $roomModel->GCSendAddChangePlayerCar($id);
        }
        return true;
    }

    // 玩家修改状态,更新等级连胜等数据
    public function updatePlayerState(int $playerId, int $roomID, $state): bool {
        /**
         * @var RoomModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change player state search room error!', (array)$room
            );
            $roomModel->GCChangePlayerState(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }
        //更新玩家等级,段位,连胜等数据
        $showPlayer = $this->getMatchShowPlayer($this->playerId);
        if(!$roomModel->updatePlayerShowAndState($room, $showPlayer, $state)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update player state error!', (array)$room
            );
            $roomModel->GCChangePlayerState(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }
        $roomModel->GCJoinRoom(ClientErrorCode::CLIENT_SUCCESS, $room);
        $roomModel->GCSendAddJoinRoom();
        foreach($room->playerInfo as $id => $p) {
            $roomModel->GCAddChangePlayerState(ClientErrorCode::CLIENT_SUCCESS, $this->playerID, $state, $id);
        }
        return true;
    }

    // 玩家修改状态
    public function updateCustomizePlayerState(int $roomID, $state): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change player state search customize room error!', (array)$room
            );
            $roomModel->GCChangePlayerState(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        if(!$roomModel->savePlayerState($room, $state)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update player state error!', (array)$room
            );
            $roomModel->GCChangePlayerState(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

        foreach($room->playerInfo as $id => $p) {
            $roomModel->GCAddChangePlayerState(ClientErrorCode::CLIENT_SUCCESS, $this->playerID, $state, $id);
        }
        return true;
    }

    // 修改玩家位置
    public function updatePlayerTeam(int $roomID, $team): bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();

        $room = $roomModel->newRoom($roomID);
        if(!$room->searchRoomByRoomID()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] change player team search customize room error!', (array)$room
            );
            $roomModel->GCChangePlayerTeam(ClientErrorCode::ERROR_ROOM_NOT_EXIST);
            return false;
        }

        // 判断是否有人预选了这个位置
        foreach($room->playerInfo as $pid => $info) {
            if($info[RoomCustomizeData::TEAM] == $team) {
                $roomModel->GCChangePlayerTeam(ClientErrorCode::ERROR_ROOM_CHANGE_TEAM_FAILED);
                return false;
            }
        }

        if(!$roomModel->savePlayerTeam($room, $team)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[RoomCustomize-RoomInfoLogic] update player state error!', (array)$room
            );
            $roomModel->GCChangePlayerTeam(ClientErrorCode::ERROR_ROOM_OPERATE_FAILED);
            return false;
        }

        foreach($room->playerInfo as $id => $p) {
            $roomModel->GCAddChangePlayerTeam(ClientErrorCode::CLIENT_SUCCESS, $this->playerID, $team, $id);
        }
        return true;
    }

    // 收藏/取消收藏自定义房间地图
    public function saveMapID(int $mapID, int $option):bool {
        /**
         * @var RoomCustomizeModel $roomModel
         */
        $roomModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ROOM_CUSTOMIZE);
        $this->playerID = $roomModel->getPlayerID();
        /**
         * @var AccountModel $accountModel
         */
        $accountModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ACCOUNT);

        $maps = $this->getAccountProperty($this->playerID, AccountData::DB_CUSTOMIZE_ROOM_SAVE_MAP, "");
        if(empty($maps)) {
            switch($option) {
                // 1 收藏  2取消收藏
                case $accountModel::CUSTOMIZE_SAVE_MAP:
                    $mapArr = [$mapID];
                    $accountModel->saveCustomizeMap($mapArr);
                    $this->sendSaveMapIDMsg(ClientErrorCode::CLIENT_SUCCESS, $option);
                    break;
                case $accountModel::CUSTOMIZE_CANCEL_MAP:
                    $this->sendSaveMapIDMsg(ClientErrorCode::ERROR_SAVE_MAP_ERROR, $option);
                    break;
            }
        } else {
            switch($option) {
                // 1 收藏  2取消收藏
                case $accountModel::CUSTOMIZE_SAVE_MAP:
                    $mapArr = json_decode($maps, true);
                    $newMap = [];
                    if(!in_array($mapID, $mapArr)) {
                        $newMap[] = $mapID;
                        $newMap = array_merge($newMap, $mapArr);
                    }
                    $accountModel->saveCustomizeMap(array_values($newMap));
                    $this->sendSaveMapIDMsg(ClientErrorCode::CLIENT_SUCCESS, $option);
                    break;
                case $accountModel::CUSTOMIZE_CANCEL_MAP:
                    $mapArr = json_decode($maps, true);
                    foreach($mapArr as $k=>$v) {
                        if($mapID == $v) {
                            unset($mapArr[$k]);
                        }
                    }
                    $accountModel->saveCustomizeMap(array_values($mapArr));
                    $this->sendSaveMapIDMsg(ClientErrorCode::CLIENT_SUCCESS, $option);
                    break;
            }
        }
        return true;
    }

    public function sendSaveMapIDMsg(int $code, int $option) {
        $msg = new GCSaveMapID();
        $msg->setCode($code);
        $msg->setOption($option);
        SendMessage::getInstance()->sendClient(PacketId::GC_SaveMapID, $msg);
    }
}
