<?php

/**
 * pve结算目标检查
 */

namespace Game\Logic\PVE;

use Framework\Log\LogMark;
use Game\Constant\ConstTemplate\TemplateTale;
use Game\Constant\GameConstantDefine;
use Game\Constant\TaskLogicDefine;
use Game\Protobuf\MatchOperation;
use Game\Protobuf\TaleTarget;

trait PveTargetLogic
{
    //检查目标完成情况 返回[目标1完成情况(0/1),目标2,目标3]
    public function getTargetCompleteInfo(array $levelConfig, TaleTarget $taleTarget, MatchOperation $matchOperation): array
    {
        //依次检查关卡目标条件
        return array(
            $this->checkTargetIsComplete(
                $levelConfig[TemplateTale::VitalTargetType],
                $levelConfig[TemplateTale::VitalTargetCondition],
                $this->getTargetValue($taleTarget, $matchOperation, $levelConfig[TemplateTale::VitalTargetType])
            ),
            $this->checkTargetIsComplete(
                $levelConfig[TemplateTale::TargetType1],
                $levelConfig[TemplateTale::TargetCondition1],
                $this->getTargetValue($taleTarget, $matchOperation, $levelConfig[TemplateTale::TargetType1])
            ),
            $this->checkTargetIsComplete(
                $levelConfig[TemplateTale::TargetType2],
                $levelConfig[TemplateTale::TargetCondition2],
                $this->getTargetValue($taleTarget, $matchOperation, $levelConfig[TemplateTale::TargetType2])
            )
        );
    }

    //获取完成条件
    private function getTargetValue(TaleTarget $taleTarget, MatchOperation $matchOperation, int $targetType): int
    {
        $value = 0;
        //不检查的次要条件
        if ($targetType == 0) {
            return $value;
        }
        switch ($targetType)
        {
            case TaskLogicDefine::TASK_CONDITION_MATCH_COMPLETE_NUM:
                //完成比赛
                if ($taleTarget->getCostTime() > 0) {
                    $value = 1;
                }
                break;
            case TaskLogicDefine::TASK_CONDITION_MATCH_RANK_TOP_3_COMMON:
            case TaskLogicDefine::TASK_CONDITION_MATCH_RANK_TOP_2_COMMON:
            case TaskLogicDefine::TASK_CONDITION_MATCH_RANK_TOP_1_COMMON:
                //名次
                $value = $taleTarget->getRank();
                break;
            case TaskLogicDefine::TASK_CONDITION_MATCH_AI_RANGE:
                //与指定AI距离,正数为超过AI,负数为落后AI
                $value = $taleTarget->getAiRange();
                break;
            case TaskLogicDefine::TASK_CONDITION_MATCH_COMPLETE_TIME:
                //指定完成时间
                if ($taleTarget->getCostTime() > 0) {
                    $value = $taleTarget->getCostTime();
                } else {
                    $value = PHP_INT_MAX;
                }
                break;
            default:
                if (isset(GameConstantDefine::TALE_TARGET_TYPE_METHOD[$targetType])) {
                    $method = GameConstantDefine::TALE_TARGET_TYPE_METHOD[$targetType];
                    //通用操作
                    if (method_exists($matchOperation, $method)) {
                        $value = $matchOperation->$method();
                    } elseif (method_exists($taleTarget, $method)) {
                        $value = $taleTarget->$method();
                    } else {
                        LogMark::getInstance()->markDebug(
                            "[PveTargetLogic] not found method",
                            array(
                                "method" => $method
                            )
                        );
                    }
                } else {
                    LogMark::getInstance()->markDebug(
                        "[PveTargetLogic] not found targetType",
                        array(
                            "targetType" => $targetType,
                        )
                    );
                }
                break;
        }
        return $value;
    }

    /**
     * 检查条件是否完成
     * @param int $targetType   条件类型
     * @param int $targetValue  条件数值
     * @param int $value        玩家比赛数值
     * @return int  0未完成 1完成
     */
    private function checkTargetIsComplete(int $targetType, int $targetValue, int $value): int
    {
        //次要条件检查,条件类型为0,代表表里未填,不做检查,返回成功
        if ($targetType == 0) {
            return 1;
        }
        $isComp = 0;
        if (isset(GameConstantDefine::TALE_TARGET_DEAL_FUNC[$targetType])) {
            $func = GameConstantDefine::TALE_TARGET_DEAL_FUNC[$targetType];
            $isComp = $this->$func($targetValue, $value);
        } else {
            LogMark::getInstance()->markDebug(
                "[PveTargetLogic] not found target deal function"
            );
        }
        return $isComp;
    }

    //判断条件 操作值小于目标值 如:规定时间
    private function taleTargetLessThen(int $targetValue, int $value): int
    {
        return $value <= $targetValue ? 1 : 0;
    }

    private function taleTargetGreaterThen(int $targetValue, int $value): int
    {
        return $value >= $targetValue ? 1 : 0;
    }

    //与指定AI距离,正数为落后AI,负数为超过AI
    private function taleTargetAIRange(int $targetValue, int $value): int
    {
//        if ($value > 0) {
//            return 1;
//        }
//        if ($targetValue < 0) {
//            //超过AI
//            return $value <= $targetValue ? 1 : 0;
//        }
//        //不落后AI距离
        return $value <= $targetValue ? 1 : 0;
    }


    //前3名
    private function taleTargetRankIn3(int $targetValue, int $value): int
    {
        return $value <= 3 ? 1 : 0;
    }

    //前2名
    private function taleTargetRankIn2(int $targetValue, int $value): int
    {
        return $value <= 2 ? 1 : 0;
    }

    //第一名
    private function taleTargetRankIn1(int $targetValue, int $value): int
    {
        return $value <= 1 ? 1 : 0;
    }
}
