<?php

/**
 * 行驶证关卡
 */

namespace Game\Logic\PVE;


use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Framework\Network\SendMessage;
use Framework\Logic\TemplateHelp;
use Game\Constant\ClientErrorCode;
use Game\Constant\ConstTemplate\TemplateGrouping;
use Game\Constant\ConstTemplate\TemplateInitPlayer;
use Game\Constant\ConstTemplate\TemplateTale;
use Game\Constant\GameConstantDefine;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Data\PVE\LevelData;
use Game\Logic\ItemLogic;
use Game\Model\PVE\PveModel;
use Game\Protobuf\CarGroups;
use Game\Protobuf\DrivingPermitCarGroup;
use Game\Protobuf\GCDrivingPermitLevel;
use Game\Protobuf\PacketId;

trait DrivingPermitLogic
{
    use ItemLogic;
    use DrivingLicenseLogic;
    use TemplateHelp;

    public function sendDrivingPermitLevelMessage(int $carTplId)
    {
        $sendMessage = new GCDrivingPermitLevel();
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        $sendMessage->setLevel(
            $this->formatChapterAllLevelMessage(
                $pveModel->getAllPermitLevelData($carTplId)
            )
        );
        SendMessage::getInstance()->sendClient(PacketId::GC_DrivingPermitLevel, $sendMessage);
    }

    //行驶证关卡检查--进入场景和结算时检查
    public function checkDrivingPermitLevel(int $levelId): int
    {
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        $levelData = $pveModel->getLevelData($levelId);
        /**
         * @var LevelData $levelData
         */
        if ($levelData->levelId == 0) {
            return ClientErrorCode::PVE_LEVEL_UNLOCK;
        }

        return ClientErrorCode::CLIENT_SUCCESS;
    }

    //检查是否通关--行驶证关卡要求一次全部通关
    public function checkCompleteDrivingPermitLevel(LevelData $levelData, array $levelConfig, array $compTarget)
    {
        if (array_sum($compTarget) == 3) {
            //全部完成
            $levelData->target = "1-1-1";
            $levelData->status = LevelData::STATUS_OVER;
            //领取奖励
            $this->setLogAddItemSource(GameConstantDefine::ITEM_ADD_SOURCE_CAR_PERMIT_REWARD);
            foreach ($levelConfig[TemplateTale::Reward] as $rewardList) {
                $this->gainItemArr($rewardList);
            }
            $levelData->rewardStatus = "2-2-2"; //已领取
            $levelData->updateDB();
            //更新完成关卡数
            $this->addCompleteLevelNum($levelConfig[TemplateTale::Car]);
            //解锁下一关
            $this->unlockDrivingPermitLevel($levelData->levelId, $levelConfig[TemplateTale::Car]);
        } else {
            //完成部分内容
            $levelData->target = implode("-", $compTarget);
        }
        $levelData->updateDB();
        //发送关卡更新消息
        $this->sendDrivingPermitLevelMessage($levelConfig[TemplateTale::Car]);
    }

    //解锁下一关
    private function unlockDrivingPermitLevel(int $levelId, int $carTplId)
    {
        $initPlayerConfig = $this->getInitPlayerConfig();
        if (!isset($initPlayerConfig[TemplateInitPlayer::DrivingPermitLevelOrder][$carTplId])) {
            LogMark::getInstance()->markDebug(
                "[DrivingPermitLogic] not found Car level",
                array(
                    "carId" => $carTplId
                )
            );
            return;
        }
        $levelOrder = $initPlayerConfig[TemplateInitPlayer::DrivingPermitLevelOrder][$carTplId];
        if (isset($levelOrder[$levelId])) {
            //开启下一关
            /**
             * @var PveModel $pveModel
             */
            $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
            $pveModel->createLevel($levelOrder[$levelId]);
        }
    }

    //获取行驶证全部关卡
    private function addCompleteLevelNum(int $carTplId)
    {
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        $pveModel->updateCompleteLevelNum($carTplId);
    }

    public function getCarGroup(): array
    {
        $config = $this->getTable(TemplateDefine::TYPE_GROUPING);
        if (is_null($config)) {
            LogMark::getInstance()->markDebug(
                "[DrivingPermitLogic] not found Grouping config",
            );
            return array();
        }
        $carLevelNum = $this->getCarLevelNum();
        $arr = array();
        foreach ($config as $groupId => $data) {
            $drivingPermitCarGroup = new DrivingPermitCarGroup();
            $drivingPermitCarGroup->setGroupId($groupId);
            $tmp = array();
            foreach ($data[TemplateGrouping::CarGroup] as $carId) {
                $carGroup = new CarGroups();
                $carGroup->setCarId($carId);
                $carGroup->setNum($carLevelNum[$carId] ?? 0);
                $tmp[] = $carGroup;
            }
            $drivingPermitCarGroup->setCarGroup($tmp);
            $arr[] = $drivingPermitCarGroup;
        }
        return $arr;
    }

    private function getCarLevelNum(): array
    {
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        return $pveModel->getCompleteLevelNum();
    }
}
