<?php

/**
 * 驾照
 */

namespace Game\Logic\PVE;

use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Framework\Network\SendMessage;
use Game\Constant\ClientErrorCode;
use Game\Constant\ConstTemplate\TemplateInitPlayer;
use Game\Constant\ConstTemplate\TemplateLicenseReward;
use Game\Constant\ConstTemplate\TemplateTale;
use Game\Constant\GameConstantDefine;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Data\PVE\ExamData;
use Game\Data\PVE\LevelData;
use Game\Logic\ItemLogic;
use Game\Model\PVE\PveModel;
use Game\Protobuf\DrivingLicenseLevel;
use Game\Protobuf\GCDrivingLicenseChapter;
use Game\Protobuf\PacketId;

trait DrivingLicenseLogic
{
    use ItemLogic;

    //获取驾照章节信息
    public function getDrivingLicenseChapter(int $chapterId): ExamData
    {
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        return $pveModel->getChapterData($chapterId);
    }

    //章节和关卡更新消息
    public function sendChapterLevelMessage(int $chapterId)
    {
        $chapterMessage = new GCDrivingLicenseChapter();
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        $chapterMessage->setLevel(
            $this->formatChapterAllLevelMessage(
                $pveModel->getAllLevelData($chapterId)
            )
        );
        SendMessage::getInstance()->sendClient(PacketId::GC_DrivingLicenseChapter, $chapterMessage);
    }

    //整理关卡消息
    private function formatChapterAllLevelMessage(array $levelDict): array
    {
        $levelMessageList = array();
        foreach ($levelDict as $levelId => $levelData) {
            $levelMessage = new DrivingLicenseLevel();
            $levelMessage->setId($levelData->levelId);
            $levelMessage->setStatus($levelData->status);
            //奖励状态
            $tmp = explode("-",$levelData->rewardStatus);
            $tNum = array_sum($tmp);
            $rewardStatus = LevelData::REWARD_STATUS_CAN_NOT_GET;
            if ($tNum == 3) {
                $rewardStatus = LevelData::REWARD_STATUS_CAN_GET;
            } elseif ($tNum == 6) {
                $rewardStatus = LevelData::REWARD_STATUS_ALREADY_GET;
            }
            $levelMessage->setRewardStatus($rewardStatus);
            //目标状态
            $tmp = explode("-",$levelData->target);
            $levelMessage->setTargetStatus($tmp);
            $levelMessageList[] = $levelMessage;
        }
        return $levelMessageList;
    }

    //驾照关卡检查--进入场景和结算时检查
    public function checkDrivingLicenseLevel(int $levelId): int
    {
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        $levelData =  $pveModel->getLevelData($levelId);
        /**
         * @var LevelData $levelData
         */
        if ($levelData->levelId == 0) {
            return ClientErrorCode::PVE_LEVEL_UNLOCK;
        }

        return ClientErrorCode::CLIENT_SUCCESS;
    }

    //检查是否通关--驾照关卡要求一次全部通关
    public function checkCompleteDrivingLicenseLevel(LevelData $levelData, array $levelConfig, array $compTarget)
    {
        if (array_sum($compTarget) == 3) {
            //全部完成
            $levelData->target = "1-1-1";
            $levelData->status = LevelData::STATUS_OVER;
            //领取奖励
            $this->setLogAddItemSource(GameConstantDefine::ITEM_ADD_SOURCE_CAR_LICENSE_REWARD);
            foreach ($levelConfig[TemplateTale::Reward] as $rewardList) {
                $this->gainItemArr($rewardList);
            }
            $levelData->rewardStatus = "2-2-2"; //已领取
            $levelData->updateDB();
            //通关后检查
            $this->checkUpdateDrivingLicense($levelConfig[TemplateTale::Chapters]);
        } else {
            //完成部分内容
            $levelData->target = implode("-", $compTarget);
        }
        $levelData->updateDB();
        //发送关卡更新消息
        $this->sendChapterLevelMessage($levelConfig[TemplateTale::Chapters]);
    }

    //通关之后 检查是否更新驾照,开启新关卡,章节
    private function checkUpdateDrivingLicense(int $chapterId)
    {
        //获取章节信息
        $chapterData = $this->getDrivingLicenseChapter($chapterId);
        ++$chapterData->examSchedule;
        //检查是否完成全部关卡
        $initPlayerConfig = $this->getInitPlayerConfig();
        $levelList = $initPlayerConfig[TemplateInitPlayer::DrivingLicenseLevel][$chapterId];
        if ($chapterData->examSchedule >= count($levelList)) {
            //通关全部关卡 更新大奖状态 更新驾照 开启新章节 解锁全部关卡
            $chapterData->drivingLicenseDate = Utils::getServerTimestamp();
            $chapterData->updateDB();
            $this->updateDrivingLicense($chapterId);
            $this->unlockNextChapter($chapterId + 1, $initPlayerConfig);
            //领取大奖奖励
            $this->getChapterReward($chapterId);
        } else {
            $chapterData->updateDB();
        }
    }

    //更新驾照信息
    private function updateDrivingLicense(int $chapterId)
    {
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        $pveModel->makeDrivingLicense($chapterId);
    }

    //开启新章节
    private function unlockNextChapter(int $chapterId, array $initPlayerConfig)
    {
        if (!isset($initPlayerConfig[TemplateInitPlayer::DrivingLicenseLevel][$chapterId])) {
            LogMark::getInstance()->markDebug(
                "[DrivingLicenseLogic] unlock new chapter error",
                array(
                    "chapterId" => $chapterId
                )
            );
            return;
        }
        /**
         * @var PveModel $pveModel
         */
        $pveModel = ModelManager::getInstance()->getModel(ModelTypeDefine::PVE);
        $pveModel->createChapter($chapterId);
        //解锁全部关卡
        foreach ($initPlayerConfig[TemplateInitPlayer::DrivingLicenseLevel][$chapterId] as $levelId) {
            $pveModel->createLevel($levelId);
        }
    }

    private function getInitPlayerConfig(): array
    {
        return $this->getTitle(TemplateDefine::TYPE_INIT_PLAYER, TemplateInitPlayer::DEFAULT_ID);
    }

    private function getChapterReward(int $chapterId)
    {
        $config = $this->getTitle(TemplateDefine::TYPE_LICENSE_REWARD, $chapterId);
        if (is_null($config)) {
            return;
        }
        $this->gainItemArr($config[TemplateLicenseReward::Reward]);
    }
}
