<?php

/**
 * 跨天重置
 */

namespace Game\Logic;

use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Game\Config\GameConfig;
use Game\Constant\EventTypeDefine;
use Game\Constant\ModelTypeDefine;
use Game\Data\AccountData;
use Game\Data\Club\ClubPlayerData;
use Game\Data\EventConditionData;
use Game\Data\FriendIntimacyRewardData;
use Game\Logic\Task\DailyTaskLogic;
use Game\Model\AccountModel;
use Game\Model\Activity\ScuffleModel;
use Game\Model\Club\ClubModel;
use Game\Model\EventConditionModel;
use Game\Model\Friend\FriendIntimacyMarkModel;
use Game\Model\Friend\FriendModel;
use Game\Model\MapStatisticsModel;
use Game\Model\PaiweiLevelModel;
use Game\Model\Shop\ShopItemModel;
use Game\Model\Shop\ShopSeasonModel;
use Game\Model\Shop\ShopZLModel;
use Game\Model\Task\Manager\FriendIntimacyTaskManager;

trait CrossDayResetLogic
{
    use DailyTaskLogic;

    public function crossDayReset(int $lastTime)
    {
        //跨天重置
        if (Utils::checkIsCrossDay($lastTime)) {
            LogMark::getInstance()->markDebug("[CrossDayResetLogic] start cross day reset");
            $this->resetOnlineTime();
            $this->resetDailyTask();
            $this->updateLoginDate();
            $this->resetClubPlayerDailyDonate();
            $this->resetShopDailyLimit();
            $this->resetScuffleDailyData();
            $this->resetFriendIntimacyTask();
            // 触发任务
            $this->triggerEvent(EventTypeDefine::EVENT_TYPE_TASK_LOGIN_DAY, []);

            // 跨天检测福利任务
            $this->checkWelfareLogin();
            // 检测限时活动
            $this->checkFirstLoginLimitActivity();
            // 检测目标活动
            $this->checkTargetLimitActivity();
            // 检测节日活动
            $this->checkFestivalLimitActivity();
        }
        //跨周重置
        if (Utils::checkIsCrossWeek($lastTime)) {
            LogMark::getInstance()->markDebug("[CrossDayResetLogic] start cross week reset");
            $this->resetFriendIntimacyData();
            $this->resetMapStatisticsData(MapStatisticsModel::TITLE_WEEK);
        }
        //跨月重置
        if (Utils::checkIsCrossMonth($lastTime)) {
            LogMark::getInstance()->markDebug("[CrossDayResetLogic] start cross month reset");
            $this->resetMapStatisticsData(MapStatisticsModel::TITLE_MONTH);
        }
    }

    //重置在线时长
    private function resetOnlineTime()
    {
        /**
         * @var EventConditionModel $condition
         */
        $condition = ModelManager::getInstance()->getModel(ModelTypeDefine::EVENT_CONDITION);
        $condition->resetEventConditionValue(EventConditionData::EVENT_CONDITION_KEY_ONLINE_TIME);
    }

    //跨天更新登录天数
    private function updateLoginDate()
    {
        /**
         * @var AccountModel $accountModel
         */
        $accountModel = ModelManager::getInstance()->getModel(ModelTypeDefine::ACCOUNT);
        if ($accountModel->search($this->playerId, $playerInfo)) {
            $day = date("Ymd") - date("Ymd", intval($playerInfo[AccountData::DB_LOGIN_TIME]));
            if ($day > 0) {
                $loginNum = isset($playerInfo[AccountData::DB_LOGIN_NUM]) ?
                    (int)$playerInfo[AccountData::DB_LOGIN_NUM] + 1 :
                    2;
                $loginContinuousNum = isset($playerInfo[AccountData::DB_LOGIN_CONTINUOUS_NUM]) ?
                    (int)$playerInfo[AccountData::DB_LOGIN_CONTINUOUS_NUM] + 1 :
                    2;
                $accountModel->updatePlayerLoginDay($loginNum, $loginContinuousNum, Utils::getServerTimestamp());
            }
        }
    }

    //重置俱乐部玩家每日捐献
    private function resetClubPlayerDailyDonate()
    {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        $clubPlayerData = new ClubPlayerData($clubModel->playerId, GameConfig::getInstance()->SERVER_ID());
        if ($clubModel->searchClubPlayerData($clubPlayerData)) {
            /**
             * @var ClubPlayerData $clubPlayerData
             */
            $clubPlayerData->setDailyDonateFlag(ClubPlayerData::DAILY_DONATE_STATUS_DOING);
            $clubPlayerData->saveDB();
        }
    }

    //重置商店每日限购
    private function resetShopDailyLimit()
    {
        //道具商城每日限购
        /**
         * @var ShopItemModel $shopItemModel
         */
        $shopItemModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SHOP_ITEM);
        $shopItemModel->resetDailyLimit();
        //赛季排位商城每日限购
        /**
         * @var ShopSeasonModel $shopSeasonModel
         */
        $shopSeasonModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SHOP_SEASON);
        $shopSeasonModel->resetDailyLimit();
        //战令商城每日限购
        /**
         * @var ShopZLModel $zlModel
         */
        $zlModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SHOP_ZL);
        $zlModel->resetDailyLimit();
    }

    //重置大乱斗每日数据
    private function resetScuffleDailyData()
    {
        /**
         * @var ScuffleModel $scuffleModel
         */
        $scuffleModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SCUFFLE);
        $scuffleModel->dailyReset();
    }

    //重置好友亲密度等
    private function resetFriendIntimacyData()
    {
        //TODO:有问题 每个人需要清空所有好友的数据?

        //检查删除小火苗
        /**
         * @var FriendModel $friend
         */
        $friend = ModelManager::getInstance()->getModel(ModelTypeDefine::FRIEND);
        $friend->resetIntimacyFire();
        //清空比赛 送礼记录
        /**
         * @var FriendIntimacyMarkModel $friendIntimacyMarkModel
         */
        $friendIntimacyMarkModel = ModelManager::getInstance()->getModel(ModelTypeDefine::FRIEND_INTIMACY_MARK);
        $friendIntimacyMarkModel->clearMarkData();
        //检查并重置活动
        /**
         * @var FriendIntimacyTaskManager $friendIntimacyTaskManager
         */
        $friendIntimacyTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_FRIEND_INTIMACY_REWARD);
        $data = $friendIntimacyTaskManager->getFriendIntimacySchedule();
        if ($data->friendIntimacyRewardOpen >= FriendIntimacyRewardData::OPEN) {
            $friendIntimacyTaskManager->resetFriendIntimacySchedule();
        }
    }

    //重置好友亲密度任务
    private function resetFriendIntimacyTask()
    {
        //检查并重置活动
        /**
         * @var FriendIntimacyTaskManager $friendIntimacyTaskManager
         */
        $friendIntimacyTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_FRIEND_INTIMACY_REWARD);
        $data = $friendIntimacyTaskManager->getFriendIntimacySchedule();
        if ($data->friendIntimacyRewardOpen >= FriendIntimacyRewardData::OPEN) {
            $friendIntimacyTaskManager->resetFriendIntimacyTask();
        }
    }

    //重置地图统计信息
    private function resetMapStatisticsData(int $titleType)
    {
        /**
         * @var MapStatisticsModel $mapStatisticsModel
         */
        $mapStatisticsModel = ModelManager::getInstance()->getModel(ModelTypeDefine::MAP_STATISTICS);
        $mapStatisticsModel->clearMapStatisticsDataByTitleType($titleType);
    }
}

