<?php
/**
 * 俱乐部逻辑类
 */

namespace Game\Logic\Club;

use DateTimeImmutable;
use DateTimeZone;
use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Framework\DB\Handler\PlayerDBHandler;
use Framework\Lib\Utils;
use Framework\Logic\RankLogic;
use Game\Config\GameConfig;
use Game\Constant\ClientErrorCode;
use Game\Constant\ConstTemplate\TemplateClubStore;
use Game\Constant\ConstTemplate\TemplateClubLevel;
use Game\Constant\ConstTemplate\TemplateClubScheduleReward;
use Game\Constant\GameConstantDefine;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Framework\Logic\TemplateHelp;
use Game\Constant\TemplateDefine;
use Game\Data\AccountData;
use Game\Data\Car\CarData;
use Game\Data\Club\BattlePassData;
use Game\Data\Club\ClubData;
use Game\Data\Club\ClubPlayerData;
use Game\Data\MoneyData;
use Game\Logic\CarLogic;
use Game\Logic\ItemLogic;
use Game\Logic\MoneyLogic;
use Game\Logic\NoticeLogic;
use Game\Model\Club\ClubModel;
use Game\Protobuf\Club;

trait ClubLogic {

    use PlayerDBHandler;
    use TemplateHelp;
    use MoneyLogic;
    use CarLogic;
    use ItemLogic;
    use MoneyLogic;
    use RankLogic;
    use NoticeLogic;

    public int $playerID;

    // 创建俱乐部
    public function createClub(int $image, int $bgImage, string $name, array $subMoney): bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);

        // 判断昵称是否重复
        if($clubModel->clubNameCheck($name)) {
            $clubModel->GCJoinClub(ClientErrorCode::ERROR_NAME_EXIST);
            $clubModel->GCSendJoinClub();
            return false;
        }

        if(!$clubModel->createClub($image, $bgImage, $name, $subMoney)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] create club error!',
                ['image' => $image, 'bgImage' => $bgImage, 'name' => $name, 'subMoney' => $subMoney]
            );
            return false;
        }
        $clubModel->GCSendJoinClub();
        $this->addClubPointsRank(0, true);
        return true;
    }

    // 模糊查询俱乐部
    public function searchClubByCondition(string $search):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);

        if(empty($search)) {
            $clubModel->GCSendSearchClub(ClientErrorCode::ERROR_SEARCH_CONDITION_EMPTY);
            return false;
        }

        if(!$clubModel->searchClub($search)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] search club error!',
                ['condition' => $search]
            );
            return false;
        }
        return true;
    }

    // 申请加入俱乐部
    public function requestJoinClub(int $clubID): bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        // 判断是否存在俱乐部
        if(!$clubModel->requestJoin($clubID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] request join club error!',
                ['clubID' => $clubID]
            );
            return false;
        }
        return true;
    }

    // 修改俱乐部玩家职位
    public function changePlayerPost(int $clubID, int $playerID, int $post, int $serverID): bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if($post == $clubModel::POST_ADMIN) {
            // 成员 -> 管理员 判断管理员数量
            if($clubModel->checkAdminIsMax($clubID)) {
                // 发送管理员数量已满
                $clubModel->GCSendChangePlayerPost(ClientErrorCode::ERROR_ADMIN_NUM_IS_MAX);
                return true;
            }
        }

        if(!$clubModel->changePlayerPost($playerID, $serverID,$post)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[ClubLogic] change club player post error!',
                ['clubID' => $clubID, 'playerID' => $playerID, 'post' => $post]
            );
            $clubModel->GCSendChangePlayerPost(ClientErrorCode::ERROR_EXEC_FAILED);
        }
        $clubModel->GCSendChangePlayerPost(ClientErrorCode::CLIENT_SUCCESS);
        return false;
    }

    // 修改俱乐部基础配置
    public function changeClub(int $clubID, int $type, int $params):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->updateClub($clubID, $type, $params)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] create club error!',
                ['clubID' => $clubID, 'type' => $type, 'params' => $params]
            );
            return false;
        }
        return true;
    }

    // 修改俱乐部宣言
    public function changeClubNotice(int $clubID, string $notice):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->updateClubNotice($clubID, $notice)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] create club error!',
                ['clubID' => $clubID, 'notice' => $notice]
            );
            return false;
        }
        return true;
    }

    // 更换队长
    public function replaceClubMaster(int $clubID, int $playerID, int $serverID): bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->updateClubMaster($clubID, $playerID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] replace master error!',
                ['clubID' => $clubID, 'playerID' => $playerID]
            );
            return false;
        }
        if(!$clubModel->changePlayerPost($playerID, $serverID, $clubModel::POST_LEADER)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] change player post error!',
                ['clubID' => $clubID, 'playerID' => $playerID]
            );
            $clubModel->GCSendReplaceMaster(ClientErrorCode::ERROR_CLUB_CHANGE_ERROR);
            return false;
        }
        $clubModel->GCSendReplaceMaster(ClientErrorCode::CLIENT_SUCCESS);
        return true;
    }

    // 修改俱乐部标签
    public function changeTag(int $clubID, array $tag):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->updateClubTag($clubID, $tag)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] create club error!',
                ['clubID' => $clubID, 'tag' => $tag]
            );
            return false;
        }
        return true;
    }

    // 获取俱乐部信息
    public function getClubInfo(int $clubID):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->getClubInfo($clubID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[ClubLogic] get club info by clubID error!',
                ['clubID' => $clubID]
            );
            return false;
        }
        $clubModel->GCSendGetClubInfo();
        return true;
    }

    // 审核加入申请
    public function approvalJoin(int $clubID, int $playerID, bool $pass, string $name, int $serverID): bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);

        // 拒绝加入
        if(!$pass) {
            if(!$clubModel->delClubJoinData($clubID, $playerID)) {
                LogMark::getInstance()->markError(
                    GameErrorCode::DATA_UPDATE_ERROR,
                    '[ClubLogic] [pass] player join club exec error!',
                    ['clubID' => $clubID]
                );
                $clubModel->GCSendApprovalJoin(ClientErrorCode::ERROR_EXEC_FAILED, $pass, GameConfig::getInstance()->SERVER_ID());
                return false;
            }
            $clubModel->GCSendApprovalJoin(ClientErrorCode::CLIENT_SUCCESS, $pass, GameConfig::getInstance()->SERVER_ID());
            $clubModel->GCSendApprovalJoin(ClientErrorCode::ERROR_REJECT_JOIN, $pass, $serverID, $playerID);
            return true;
        }
        // 俱乐部是否满员
        if($clubModel->clubMemberIsMax($clubID)) {
            $clubModel->GCSendApprovalJoin(ClientErrorCode::ERROR_ADMIN_NUM_IS_MAX, $pass, GameConfig::getInstance()->SERVER_ID());
            return true;
        }

        // 玩家是否拥有俱乐部
        if($clubModel->playerInClub($playerID, $serverID)) {
            if(!$clubModel->delClubJoinData($clubID, $playerID)) {
                LogMark::getInstance()->markError(
                    GameErrorCode::DATA_UPDATE_ERROR,
                    '[ClubLogic] [member max] player join club exec error!',
                    ['clubID' => $clubID]
                );
                $clubModel->GCSendApprovalJoin(ClientErrorCode::ERROR_EXEC_FAILED, $pass, GameConfig::getInstance()->SERVER_ID());
                return false;
            }
            $clubModel->GCSendApprovalJoin(ClientErrorCode::ERROR_PLAYER_IN_CLUB, $pass, GameConfig::getInstance()->SERVER_ID());
            return true;
        }

        if(!$clubModel->passApprovalJoin($clubID, $playerID, $serverID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[ClubLogic] player join club error!',
                ['clubID' => $clubID]
            );
            $clubModel->GCSendApprovalJoin(ClientErrorCode::ERROR_EXEC_FAILED, $pass, GameConfig::getInstance()->SERVER_ID());
            return false;
        }
        $clubModel->GCSendApprovalJoin(ClientErrorCode::CLIENT_SUCCESS, $pass, GameConfig::getInstance()->SERVER_ID());
        $clubModel->GCSendApprovalJoin(ClientErrorCode::CLIENT_SUCCESS, $pass, $serverID, $playerID);
        $this->updateClubPlayerNotice(ClubPlayerData::CLUB_STATUS_ADD);
        return true;
    }

    // 获取俱乐部剩余席位
    public function findClubVacancySeat(ClubData $c): int {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        return $clubModel->getClubVacancySeat($c);
    }

    // 获取自己俱乐部id
    public function findMyClubID():bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->findMyClubID()) {
//            LogMark::getInstance()->markError(
//                GameErrorCode::DATA_SEARCH_ERROR,
//                '[ClubLogic] find my clubID error!',
//            );
            return false;
        }
        return true;
    }

    // 弹劾队长
    public function impeachMaster(int $clubID, int $serverID):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->impeachMaster($clubID, $serverID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[ClubLogic] impeach master error!',
            );
            $clubModel->GCSendImpeachMaster(ClientErrorCode::ERROR_EXEC_FAILED);
            return false;
        }
        return true;
    }

    // 清空审核
    public function clearApplyJoinList(int $clubID): bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->clearApplyJoinList($clubID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_DELETE_ERROR,
                '[ClubLogic] clear apply join list error!',
                ['clubID' => $clubID]
            );
            return false;
        }
        return true;
    }

    // 删除成员
    public function delClubMember(int $clubID, int $playerID, int $serverID): bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);

        // 玩家是否在俱乐部中
        if(!$clubModel->playerAndMeInClub($playerID, $serverID)) {
            $clubModel->GCSendDelClubMember(ClientErrorCode::ERROR_MEMBER_NOT_EXIST);
            return true;
        }

        if(!$clubModel->removeClubPlayer($clubID, $playerID, $serverID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[ClubLogic] player join club error!',
                ['clubID' => $clubID]
            );
            $clubModel->GCSendDelClubMember(ClientErrorCode::ERROR_EXEC_FAILED);
            return false;
        }
        $clubModel->GCSendDelClubMember(ClientErrorCode::CLIENT_SUCCESS);
        $clubModel->GCAddDelClubMember(ClientErrorCode::CLIENT_SUCCESS, $playerID);
        return true;
    }

    // 退出俱乐部
    public function exitClub():bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->exitClub()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_DELETE_ERROR,
                '[ClubLogic] player exit club error!',
            );
            $clubModel->GCSendExitClub(ClientErrorCode::ERROR_EXEC_FAILED);
            return false;
        }
        $clubModel->GCSendExitClub(ClientErrorCode::CLIENT_SUCCESS);
        return true;
    }

    // 解散俱乐部
    public function disbandClub(int $club):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if(!$clubModel->disbandClub($club)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_DELETE_ERROR,
                '[ClubLogic] player disband club error!',
            );
            $clubModel->GCSendDisbandClub(ClientErrorCode::ERROR_EXEC_FAILED);
            return false;
        }
        $clubModel->GCSendDisbandClub(ClientErrorCode::CLIENT_SUCCESS);
        return true;
    }

    // 判断此玩家是否与自己一个俱乐部
    public function playerAndMeInClub(int $playerID, int $serverID):bool {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);

        return $clubModel->playerAndMeInClub($playerID, $serverID);
    }

    //随机搜索俱乐部
    public function randomClubList(): array
    {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if (!$clubModel->searchRandomClubList($lists)){
            return [];
        }
        return $this->getClubsData($lists);
    }

    //查询俱乐部列表的数据
    public function getClubsData(array $lists): array
    {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        $clubList = [];
        foreach ($lists as $id){
            $clubList[] = $clubModel->searchClubData($id);
        }
        return $clubList;
    }

    public function sendClientData(array $dataList): array
    {
        $Info = [];
        foreach ($dataList as $index => $data){
            $club = new Club();
            $club->setClubID($data[ClubData::CLUB_ID]);
            $club->setClubName($data[ClubData::CLUB_NAME]);
            $club->setClubMaster($data[AccountData::DB_NICK_NAME]);
            $club->setImage($data[ClubData::IMAGE]);
            $club->setBgImage($data[ClubData::BG_IMAGE]);
            $club->setPlayers($data[ClubData::PLAYER_INFO]);
            $club->setLevelScore($data[ClubData::LEVEL_SCORE]);
            $club->setClubNotice($data[ClubData::CLUB_NOTICE]);
            $club->setClubTags($data[ClubData::CLUB_TAGS]);
            $club->setLevelLimit($data[ClubData::LEVEL_LIMIT]);
            $club->setRankdLimit($data[ClubData::RANK_LIMIT]);
            $club->setJoinState($data[ClubData::JOIN_STATE]);
            $club->setClubLevel($data[ClubData::LEVEL]);
            $club->setCarsPerformance($data[AccountData::DB_CARS_PERFORMANCE_SCORE]);
            $club->setIndex($index);
            $Info[] = $club;
        }
        return $Info;
    }

    //更新活跃度
    public function upActivity($activityScore)
    {
        if ($activityScore <= 0){
            return true;
        }
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if ($clubModel->upActive((int)$activityScore)){
            return false;
        }
        return true;
    }

    //俱乐部广场
    public function getClubSquare(&$ret): int
    {
        $clubData = $this->getOneClubData();
        if (is_int($clubData)){
            return false;
        }

        // 俱乐部荣誉

        // 舞池 前三名玩家的最好的车展示[车是展示俱乐部用车,还是自有车辆]

    }

    public function createClubCar($carId)
    {
        //创建有时间限制的Car
        // 星期一 车辆失效时间
        $refreshTime = Utils::makeImmutableTime("next Monday");  //俱乐部刷新时间
        $useTime = Utils::makeImmutableTime("next Day");  //车辆使用时间
        if ($useTime > $refreshTime){
            $useTime = $refreshTime;
        }
        //根据俱乐部等级, 模板查 车辆类型Id, 等级;
        $carLevel = 1;

        $this->createCar($carId, $carLevel, $refreshTime);

        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        $clubModel->upPlayerData([ClubPlayerData::CARS_REFRESH => $refreshTime]);
    }

    //俱乐部广场进入验证
    public function match()
    {
        //检测是否需要重置
        //检测车辆使用时间
        //如果是比赛,还需要检测 是否是练习赛
    }

    //查看俱乐部中我的车辆信息
    public function getClubCarsInfo(&$data): int
    {
        if (!$this->getOneClubPlayerData($clubPlayerData)){
            return ClientErrorCode::ERROR_CLUB_PLAYER_NOT_JOIN;
        }
        if ($clubPlayerData->useCars != ""){
            $cars = json_decode($clubPlayerData->useCars, true);
            if (is_array($cars) && count($cars) > 0){
                foreach ($cars as $carId => $carCutOffTime){
                    /**
                     * @var $carData CarData
                     */
                    $carData = $this->searchCarDataByCarID($carId);
                    $data[] = array(
                        CarData::UID => $carData->uid,
                        CarData::CAR_ID => $carData->carID,
                        CarData::LEVEL => $carData->level,
                        CarData::ACTIVE_SKILL => $carData->activeSkill,
                        CarData::PASSIVE_SKILL => $carData->passiveSkill,
                        CarData::PROFICIENCY => $carData->proficiency,
                        CarData::PERFORMANCE => $carData->performance,
                        "CutOffTime" => $carCutOffTime,
                    );
                }
            }
        }else{
            $data = "";
        }

        return ClientErrorCode::CLIENT_SUCCESS;
    }
    //领取俱乐部免费车辆
    public function drawClubCar($carId): bool
    {
        if (!$this->getOneClubPlayerData($clubPlayerData)){
            return ClientErrorCode::ERROR_CLUB_PLAYER_NOT_JOIN;
        }
        $now = Utils::makeImmutableTime("now");
        if ($clubPlayerData->useCars != "") {
            $cars = json_decode($clubPlayerData->useCars, true);
            $carNum = count($cars);
            if ($carNum >= 3){
                return ClientErrorCode::ERROR_CLUB_CARS_MAX;
            }
            if ($carNum != 0){
                //检测有车辆仍未到期,可以使用
                foreach ($cars as $car => $time){
                    if ($time > $now){
                        return ClientErrorCode::ERROR_CLUB_CARS_NOW_USE;
                    }
                }
                //检测车辆
                //创建车辆
            }


        }
    }

    //俱乐部每个玩家车辆重置
    public function resetCars()
    {
        if (!$this->getOneClubPlayerData($clubPlayerData)){
            return false;
        }
        if (Utils::getServerTimestamp() < $clubPlayerData->carsRefresh){
            return false;
        }

        // 创建有时间限制的Car
        // 星期一 车辆失效时间
        $refreshTime = Utils::makeImmutableTime("next Monday");
        //根据俱乐部等级, 模板查 车辆类型Id, 等级;
        $carId = 0;
        $carLevel = 1;

        $this->createCar($carId, $carLevel, $refreshTime);
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        return $clubModel->upPlayerData([ClubPlayerData::CARS_REFRESH => $refreshTime]);
    }

    //捐献功能
    public function donate($score): int
    {
        $score = abs($score);
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if (!$clubModel->donate($score, $code)){
            return $code;
        }
        return ClientErrorCode::CLIENT_SUCCESS;
    }
    // 俱乐部DJ名称设置
    public function setDJName($name): bool
    {

    }

    //俱乐部战令领取
    public function getClubBattlePassReward($type, $clubBattlePassId): int
    {
        if ($clubBattlePassId <= 0){
            return ClientErrorCode::ERROR_CLUB_BATTLE_PASS_ID;
        }
        $tplBattlePass = $this->getTitle(TemplateDefine::TYPE_CLUB_SCHEDULE_REWARD, $clubBattlePassId);
        if (is_null($tplBattlePass)){
            return ClientErrorCode::ERROR_CLUB_BATTLE_PASS_ID;
        }
        if (!$this->getOneClubPlayerData($clubPlayerData)){
            return ClientErrorCode::ERROR_CLUB_NOT_EXIST;
        }
        //判断活跃值是否符合
        if ($clubPlayerData->Active < $tplBattlePass[TemplateClubScheduleReward::Active]){
            return ClientErrorCode::ERROR_CLUB_BATTLE_ACTIVE;
        }

        if ($type == BattlePassData::Index_Club){
            //todo 检测是否有权限领取

            $rewardType = $tplBattlePass[TemplateClubScheduleReward::ClubRewardType];
            $item = [$tplBattlePass[TemplateClubScheduleReward::ClubRewardId], $tplBattlePass[TemplateClubScheduleReward::ClubRewardNum]];
        }else{
            $rewardType = $tplBattlePass[TemplateClubScheduleReward::PersonalRewardType];
            $item = [$tplBattlePass[TemplateClubScheduleReward::PersonalRewardId], $tplBattlePass[TemplateClubScheduleReward::PersonalRewardNum]];
        }

        /**
         * @var ClubModel $club
         */
        $club = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);

        //检测是否已经领取
        $battleData = $club->getBattlePass();
        if (isset($battleData[$clubBattlePassId])){
            return ClientErrorCode::ERROR_CLUB_BATTLE_RECEIVED;
        }

        if (!$club->saveBattlePass($type, $clubBattlePassId)){
            return ClientErrorCode::ERROR_CLUB_BATTLE_SAVE;
        }
        switch($rewardType){
            //增加道具
            case BattlePassData::TYPE_REWARD_ITEM:
                if (!$this->gainItem($item[0], $item[1])){
                    return ClientErrorCode::ERROR_CLUB_BATTLE_ITEM;
                }
                break;
//            //增加货币
//            case BattlePassData::TYPE_REWARD_MONEY:
//                if (!$this->addMoney($item[0], $item[1])){
//                    return ClientErrorCode::ERROR_CLUB_BATTLE_ITEM;
//                }
//                break;
            default:
                //增加经验
                if (!$club->addClubExp($item[1])){
                    return ClientErrorCode::ERROR_CLUB_BATTLE_ITEM;
                }
        }

        return ClientErrorCode::CLIENT_SUCCESS;
    }

    //获取当前battlePass数据
    public function getBattlePass(&$code)
    {
        $this->resetBattlePass($code);
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        return $clubModel->getBattlePass();
    }

    //重置俱乐部战令
    public function resetBattlePass(&$code):bool
    {
        if (!$this->getOneClubPlayerData($clubPlayerData)){
            $code = ClientErrorCode::ERROR_CLUB_NOT_EXIST;
            return false;
        }
        if ($clubPlayerData->battlePassRefresh >= Utils::getServerTimestamp()){
            $code = ClientErrorCode::CLIENT_SUCCESS;
            return true;
        }
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if (!$clubModel->resetBattlePass()){
            $code =  ClientErrorCode::ERROR_CLUB_BATTLE_RESET;
            return false;
        }
        $code = ClientErrorCode::CLIENT_SUCCESS;
        return true;
    }

    public function getBattlePassClubMoney(&$code): bool
    {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        if ($code = $clubModel->getBattlePassClubMoney($money)){
            //增加 货币
            if ($money > 0){
                $this->setLogAddItemSource(GameConstantDefine::ITEM_ADD_SOURCE_CLUB);
                $this->addMoney(MoneyData::CLUB, $money);
            }

        }
        return false;
    }

    public function getBattlePassActive(): int
    {
        if (!$this->getOneClubPlayerData($clubPlayerData)){
            return 0;
        }
        return $clubPlayerData->activeBattlePass;
    }

    //获取当前俱乐部的等级
    public function getClubLevel(): int
    {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        return $clubModel->getMineClubData(ClubData::LEVEL, 1);
    }

    public function getOneClubData()
    {
        // 我选择的车辆信息
        // 俱乐部信息
        if (!$this->getOneClubPlayerData($clubPlayer)){
            return array();
        }

        $clubId = $clubPlayer->clubID;
        if ($clubId == 0){
            return ClientErrorCode::ERROR_CLUB_NOT_EXIST;
        }
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        return $clubModel->searchClubData($clubId);
    }
    public function getOneClubPlayerData(&$clubPlayerData): bool
    {
        /**
         * @var ClubModel $clubModel
         */
        $clubModel = ModelManager::getInstance()->getModel(ModelTypeDefine::CLUB);
        $clubPlayerData = new ClubPlayerData($clubModel->playerId, GameConfig::getInstance()->SERVER_ID());
        return $clubModel->searchClubPlayerData($clubPlayerData);
    }

    //每日捐献
    public function dailyDonate(): array
    {
        /**
         * @var ClubPlayerData $clubPlayerData
         */
        //检查俱乐部玩家是否存在
        if (!$this->getOneClubPlayerData($clubPlayerData)){
            return array(
                ClientErrorCode::ERROR_CLUB_NOT_EXIST,
                $clubPlayerData->activeBattlePass,
                ClubPlayerData::DAILY_DONATE_STATUS_DONE
            );
        }
        //检查是否领取
        if ($clubPlayerData->dailyDonateFlag == ClubPlayerData::DAILY_DONATE_STATUS_DONE) {
            return array(
                ClientErrorCode::ERROR_CLUB_DAILY_DONATE_GET,
                $clubPlayerData->activeBattlePass,
                ClubPlayerData::DAILY_DONATE_STATUS_DONE
            );
        }
        //加活跃度
        //TODO:活跃度值未确定
        $addActive = 10;
        $clubPlayerData->upBattlePassActive($addActive);
        $clubPlayerData->setDailyDonateFlag(ClubPlayerData::DAILY_DONATE_STATUS_DONE);
        $clubPlayerData->saveDB();
        return array(
            ClientErrorCode::CLIENT_SUCCESS,
            $clubPlayerData->activeBattlePass + $addActive,
            ClubPlayerData::DAILY_DONATE_STATUS_DONE
        );
    }
}
