<?php

/**
 * 福利活动信息
 */

namespace Game\Logic\Activity;


use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Game\Constant\EventTypeDefine;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Data\AccountData;
use Game\Model\Activity\WelfareModel;
use Game\Model\Task\Manager\WeekTaskManager;
use Framework\Logic\TimerLogic;

trait WelfareLogic
{
    use TimerLogic;
    // 跨天检测
    public function checkWelfareLogin(): void
    {
        // 判断增加新登录
        $this->change7NewData();
        // 重置前一日记录
        $this->resetLoginData();
        // 处理周签任务 与 结算跨周数据
        $this->completeWeekTask();
        // 重置每周登录活动
        $this->resetWeekLoginData();
        // 每周登录是否是周末
        $this->checkWeekLoginData();

        // 触发周任务 周六日累计登录
        $this->triggerEvent(EventTypeDefine::EVENT_TYPE_TASK_WEEK_LOGIN, []);
        // 跨天触发
        $this->triggerEvent(EventTypeDefine::EVENT_TYPE_TASK_DAY_LOGIN, []);

    }

    private function change7NewData() {
        // 处理福利活动逻辑
        /**
         * @var WelfareModel $welfareModel
         */
        $welfareModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WELFARE);
        if(!$welfareModel->data->searchDB()) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_SEARCH_ERROR,
                'login 7 new activity data error!'
            );
            return ;
        }
        $createTime = $this->getAccountAttribute(AccountData::DB_CREATE_TIME);
        if($createTime + $welfareModel::ACTIVITY_7NEW_TIME_LIMIT <= Utils::getServerTimestamp()) {
            return ;
        }
        if($welfareModel->data->newNumber >= 7) {
            return ;
        }
        ++$welfareModel->data->newNumber;
        $welfareModel->data->updateDB();
    }

    private function resetLoginData() {
        // 处理福利活动逻辑
        /**
         * @var WelfareModel $welfareModel
         */
        $welfareModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WELFARE);
        $welfareModel->resetLoginRecord();
    }

    public function addLoginActivity(): void
    {
        // 增加每日登录活动 timer
        $time = strtotime(date('Y-m-d'));
        $timeStart1 = $time + WelfareModel::NOON_START * 60 * 60;
        $timeStart2 = $time + WelfareModel::NIGHT_START * 60 * 60;
        $this->addTimer(ModelTypeDefine::WELFARE, $this->playerId, $timeStart1);
        $this->addTimer(ModelTypeDefine::WELFARE, $this->playerId, $timeStart2);
    }

    public function changeLoginData() {
        /**
         * @var WelfareModel $welfareModel
         */
        $welfareModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WELFARE);
        $hour = date("H");
        $reward = 0;
        if($hour >= $welfareModel::NOON_START && $hour <= $welfareModel::NOON_END) {
            $reward = $welfareModel::NOON;
        }
        if($hour >= $welfareModel::NIGHT_START && $hour <= $welfareModel::NIGHT_END) {
            $reward = $welfareModel::NIGHT;
        }
        if($reward == 0) {
            return ;
        }
        $welfareModel->data->searchDB();
        if(!in_array($reward, $welfareModel->data->loginRecord1)) {
            $welfareModel->data->loginRecord1[] = $reward;
            $welfareModel->data->updateDB();
        }
    }

    private function completeWeekTask() {
        /**
         * @var WelfareModel $welfareModel
         */
        $welfareModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WELFARE);

        /**
         * @var WeekTaskManager $weekTaskManager
         */
        $weekTaskManager = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_WELFARE_WEEK);

        $loginTime = $this->getAccountAttribute(AccountData::DB_LOGIN_TIME);

        if($this->checkCrossWeek($loginTime)) {
            // 清除周签记录
            $welfareModel->resetWeekRecord();
            // 重置周任务
            $weekTaskManager->resetWelfareWeekTask();
        }

        // 记录周活动登录记录
        $week = date("w");
        $welfareModel->data->searchDB();
        if(!in_array($week, $welfareModel->data->weekRecord)) {
            $welfareModel->data->weekRecord[] = (int)$week;
        }
        $welfareModel->data->updateDB();

        // 首次登录任务事件开启关闭
        $weekTaskManager->updateWelfareWeekTaskOnDayLogin();
    }

    private function resetWeekLoginData() {
        /**
         * @var WelfareModel $welfareModel
         */
        $welfareModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WELFARE);

        $loginTime = $this->getAccountAttribute(AccountData::DB_LOGIN_TIME);

        if($this->checkCrossWeek($loginTime)) {
            // 重置周登录状态
            $welfareModel->resetWeekLoginRecord();
        }
    }

    private function checkWeekLoginData() {
        /**
         * @var WelfareModel $welfareModel
         */
        $welfareModel = ModelManager::getInstance()->getModel(ModelTypeDefine::WELFARE);
        $welfareModel->data->searchDB();
        if($welfareModel->data->weekLoginState !== $welfareModel::STATE_NOT_REWARD) {
            return ;
        }
        if($this->checkWeekDown()) {
            // 重置周登录状态
            $welfareModel->data->weekLoginState = $welfareModel::STATE_REWARD;
            $welfareModel->data->updateDB();
        }
    }

    // 跨周检测
    private function checkCrossWeek(int $time): bool
    {
        if ($time == 0) {
            return false;
        }
        return date("W", $time) !== date("W");
    }

    // 判断是否是周末
    private function checkWeekDown(): bool {
        // 周六周日 可以领取奖励
        if(in_array(date('w'), [0, 6])) {
//            if(in_array(date('w'), [1, 2])) {
            return true;
        }
        return false;
    }
}
