<?php

/**
 * 限时活动
 */

namespace Game\Logic\Activity;


use Framework\DB\Handler\PlayerDBPipe;
use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\MVC\ModelManager;
use Game\Constant\ConstTemplate\TemplateTask;
use Game\Constant\ConstTemplate\TemplateTaskExchange;
use Game\Constant\ConstTemplate\TemplateTaskFestival;
use Game\Constant\ConstTemplate\TemplateTaskTarget;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Data\AccountData;
use Game\Model\Activity\LimitTimeModel;
use Framework\Logic\TemplateHelp;
use Game\Model\Activity\TaskExchangeLoginModel;
use Game\Model\Shop\ShopTaskExchangeModel;
use Game\Model\Task\Manager\TaskMangerModelBase;

trait LimitActivityLogic
{
    use TemplateHelp;

    public function checkFirstLoginLimitActivity(): void
    {
        /**
         * @var LimitTimeModel $limitTimeModel
         */
        $limitTimeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::LIMIT_TIME_ACTIVITY);
        $limitTimeModel->getLimitTimeData();

        $taskExchange = $this->getTable(TemplateDefine::TYPE_TASK_EXCHANGE);
        if(is_null($taskExchange)) {
            LogMark::getInstance()->markError(GameErrorCode::DATA_SEARCH_ERROR,
            '[limitActivityLogic] get task exchange data error!');
            return ;
        }

        $nowTime = Utils::getServerTimestamp();
        foreach ($taskExchange as $task) {
            $taskID = $task[TemplateTaskExchange::Id];

            /**
             * @var TaskExchangeLoginModel $taskExchangeLoginModel
             */
            $taskExchangeLoginModel = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_EXCHANGE_LOGIN);
            $taskExchangeLoginModel->data->searchDB($taskID);

            /**
             * @var LimitTimeModel $limitTimeModel
             */
            $limitTimeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::LIMIT_TIME_ACTIVITY);
            $limitTimeModel->data->searchDB();

            // 检测用户开启的活动
            if($nowTime >= $task[TemplateTaskExchange::StartTime] && $nowTime
                <= $task[TemplateTaskExchange::EndTime]) {
                switch ($task[TemplateTaskExchange::TaskType]) {
                    // 累计登录
                    case TemplateTaskExchange::TASK_TYPE_LOGIN_NUM:
                        // 记录登录天数
                        $day = ceil(($nowTime - $task[TemplateTaskExchange::StartTime]) / 86400);
                        if(!in_array($day, $taskExchangeLoginModel->data->loginRecord)) {
                            $taskExchangeLoginModel->data->loginRecord[] = $day;
                            $taskExchangeLoginModel->data->updateDB($taskID);
                        }
                        break;
                    case TemplateTaskExchange::TASK_TYPE_SHOP:
                        /**
                         * @var ShopTaskExchangeModel $shopTaskExchangeModel
                         */
                        $shopTaskExchangeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::SHOP_TASK_EXCHANGE);
                        $shopTaskExchangeModel->resetShopLimit($taskID);
                        break;
                    default:
                        // 任务处理
                        if(!in_array($taskID, $limitTimeModel->data->exchangeRecord)) {
                            // 需要增加任务
                            $this->addExchangeTask($task[TemplateTaskExchange::TaskId]);
                            $limitTimeModel->data->exchangeRecord[] = $taskID;
                        } else {
                            // 需要重置任务
                            $this->updateExchangeTask($task[TemplateTaskExchange::TaskId]);
                        }
                    break;
                }

            } else {
                // 关闭累计登录
                switch ($task[TemplateTaskExchange::TaskType]) {
                    // 累计登录
                    case TemplateTaskExchange::TASK_TYPE_LOGIN_NUM:
                        if(!empty($taskExchangeLoginModel->data->loginRecord) || !empty($taskExchangeLoginModel->data->rewardRecord)) {
                            $taskExchangeLoginModel->data->loginRecord = [];
                            $taskExchangeLoginModel->data->rewardRecord = [];
                            $taskExchangeLoginModel->data->updateDB($taskID);
                        }
                        break;
                    case TemplateTaskExchange::TASK_TYPE_SHOP:
                        break;
                    default:
                        if(in_array($taskID, $limitTimeModel->data->exchangeRecord)) {
                            // 需要关闭任务
                            $this->delExchangeTask($task[TemplateTaskExchange::TaskId]);
                            $newRecord = [];
                            foreach ($limitTimeModel->data->exchangeRecord as $id) {
                                if($id != $taskID) {
                                    $newRecord[] = $id;
                                }
                            }
                            $limitTimeModel->data->exchangeRecord = $newRecord;
                        }
                        break;
                }
            }
            $limitTimeModel->data->updateDB();
        }
    }

    public function checkFestivalLimitActivity() {
        /**
         * @var LimitTimeModel $limitTimeModel
         */
        $limitTimeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::LIMIT_TIME_ACTIVITY);
        $limitTimeModel->getLimitTimeData();

        $taskFestival = $this->getTable(TemplateDefine::TYPE_TASK_FESTIVAL);
        if(is_null($taskFestival)) {
            LogMark::getInstance()->markError(GameErrorCode::DATA_SEARCH_ERROR,
                '[limitActivityLogic] get task festival data error!');
            return ;
        }

        $nowTime = Utils::getServerTimestamp();
        foreach ($taskFestival as $task) {
            $taskID = $task[TemplateTaskFestival::Id];
            /**
             * @var LimitTimeModel $limitTimeModel
             */
            $limitTimeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::LIMIT_TIME_ACTIVITY);
            $limitTimeModel->data->searchDB();

            // 检测用户开启的活动
            if($nowTime >= $task[TemplateTaskFestival::StartTime] && $nowTime
                <= $task[TemplateTaskFestival::EndTime]) {
                if(!in_array($taskID, $limitTimeModel->data->exchangeRecord)) {
                    // 需要增加任务
                    $this->addExchangeTask($task[TemplateTaskFestival::TaskId]);
                    $limitTimeModel->data->exchangeRecord[] = $taskID;
                } else {
                    // 需要重置任务
                    $this->updateExchangeTask($task[TemplateTaskFestival::TaskId]);
                }
            } else {
                // 关闭活动
                if(in_array($taskID, $limitTimeModel->data->exchangeRecord)) {
                    // 需要关闭任务
                    $this->delExchangeTask($task[TemplateTaskFestival::TaskId]);
                    $newRecord = [];
                    foreach ($limitTimeModel->data->exchangeRecord as $id) {
                        if($id != $taskID) {
                            $newRecord[] = $id;
                        }
                    }
                    $limitTimeModel->data->exchangeRecord = $newRecord;
                }
            }
            $limitTimeModel->data->updateDB();
        }
    }

    public function checkTargetLimitActivity() {
        /**
         * @var LimitTimeModel $limitTimeModel
         */
        $limitTimeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::LIMIT_TIME_ACTIVITY);
        $limitTimeModel->getLimitTimeData();

        $taskTarget = $this->getTable(TemplateDefine::TYPE_TASK_TARGET);
        if(is_null($taskTarget)) {
            LogMark::getInstance()->markError(GameErrorCode::DATA_SEARCH_ERROR,
                '[limitActivityLogic] get task target data error!');
            return ;
        }

        $nowTime = Utils::getServerTimestamp();
        foreach ($taskTarget as $task) {
            $taskID = $task[TemplateTaskTarget::Id];

            /**
             * @var TaskExchangeLoginModel $taskExchangeLoginModel
             */
            $taskExchangeLoginModel = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_EXCHANGE_LOGIN);
            $taskExchangeLoginModel->data->searchDB($taskID);

            /**
             * @var LimitTimeModel $limitTimeModel
             */
            $limitTimeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::LIMIT_TIME_ACTIVITY);
            $limitTimeModel->data->searchDB();

            // 检测用户开启的活动
            if($nowTime >= $task[TemplateTaskTarget::StartTime] && $nowTime
                <= $task[TemplateTaskTarget::EndTime]) {
                switch ($task[TemplateTaskTarget::TaskType]) {
                    // 累计登录
                    case TemplateTaskTarget::TASK_TYPE_LOGIN_NUM:
                        // 记录登录天数
                        $day = ceil(($nowTime - $task[TemplateTaskTarget::StartTime]) / 86400);
                        if(!in_array($day, $taskExchangeLoginModel->data->loginRecord)) {
                            $taskExchangeLoginModel->data->loginRecord[] = $day;
                            $taskExchangeLoginModel->data->updateDB($taskID);
                        }
                        break;
                    default:
                        // 任务处理
                        if(!in_array($taskID, $limitTimeModel->data->exchangeRecord)) {
                            // 需要增加任务
                            $this->addExchangeTask($task[TemplateTaskTarget::TaskId]);
                            $limitTimeModel->data->exchangeRecord[] = $taskID;
                        } else {
                            // 需要重置任务
                            $this->updateExchangeTask($task[TemplateTaskTarget::TaskId]);
                        }
                        break;
                }

            } else {
                // 关闭累计登录
                switch ($task[TemplateTaskTarget::TaskType]) {
                    // 累计登录
                    case TemplateTaskTarget::TASK_TYPE_LOGIN_NUM:
                        if(!empty($taskExchangeLoginModel->data->loginRecord) || !empty($taskExchangeLoginModel->data->rewardRecord)) {
                            $taskExchangeLoginModel->data->loginRecord = [];
                            $taskExchangeLoginModel->data->rewardRecord = [];
                            $taskExchangeLoginModel->data->updateDB($taskID);
                        }
                        break;
                    default:
                        if(in_array($taskID, $limitTimeModel->data->exchangeRecord)) {
                            // 需要关闭任务
                            $this->delExchangeTask($task[TemplateTaskTarget::TaskId]);
                            $newRecord = [];
                            foreach ($limitTimeModel->data->exchangeRecord as $id) {
                                if($id != $taskID) {
                                    $newRecord[] = $id;
                                }
                            }
                            $limitTimeModel->data->exchangeRecord = $newRecord;
                        }
                        break;
                }
            }
            $limitTimeModel->data->updateDB();
        }
    }

    // 注册 增加注册数据
    public function registerCheckActivity() {

        /**
         * @var TaskExchangeLoginModel $taskExchangeLoginModel
         */
        $taskExchangeLoginModel = ModelManager::getInstance()->getModel(ModelTypeDefine::TASK_EXCHANGE_LOGIN);

        /**
         * @var LimitTimeModel $limitTimeModel
         */
        $limitTimeModel = ModelManager::getInstance()->getModel(ModelTypeDefine::LIMIT_TIME_ACTIVITY);
        $limitTimeModel->getLimitTimeData();

        $taskExchange = $this->getTable(TemplateDefine::TYPE_TASK_EXCHANGE);
        if(is_null($taskExchange)) {
            LogMark::getInstance()->markError(GameErrorCode::DATA_SEARCH_ERROR,
                '[limitActivityLogic] get task exchange data error!');
            return ;
        }
        $nowTime = Utils::getServerTimestamp();

        foreach ($taskExchange as $task) {
            // 检测用户开启的活动
            if($nowTime >= $task[TemplateTaskExchange::StartTime] && $nowTime
                <= $task[TemplateTaskExchange::EndTime]) {

                // 增加任务
                $taskID = $task[TemplateTaskExchange::Id];

                switch ($task[TemplateTaskExchange::TaskType]) {
                    // 累计登录
                    case TemplateTaskExchange::TASK_TYPE_LOGIN_NUM:
                        // 记录登录天数
                        $day = ceil(($nowTime - $task[TemplateTaskExchange::StartTime]) / 86400);
                        $taskExchangeLoginModel->data->loginRecord[] = $day;
                        $taskExchangeLoginModel->data->updateDB($taskID);
                        break;
                    case TemplateTaskExchange::TASK_TYPE_SHOP:
                    default:
                        $limitTimeModel->data->exchangeRecord[] = $taskID;
                        break;
                }
            }
        }

        // 节日
        $taskFestival = $this->getTable(TemplateDefine::TYPE_TASK_FESTIVAL);
        if(is_null($taskFestival)) {
            LogMark::getInstance()->markError(GameErrorCode::DATA_SEARCH_ERROR,
                '[limitActivityLogic] get task festival data error!');
            return ;
        }
        foreach ($taskFestival as $task) {
            // 检测用户开启的活动
            if ($nowTime >= $task[TemplateTaskFestival::StartTime] && $nowTime
                <= $task[TemplateTaskFestival::EndTime]) {
                    // 需要增加任务
                $taskID = $task[TemplateTaskFestival::Id];
                $limitTimeModel->data->exchangeRecord[] = $taskID;
            }
        }

        // 目标
        $taskTarget = $this->getTable(TemplateDefine::TYPE_TASK_TARGET);
        if(is_null($taskTarget)) {
            LogMark::getInstance()->markError(GameErrorCode::DATA_SEARCH_ERROR,
                '[limitActivityLogic] get task target data error!');
            return ;
        }

        foreach ($taskTarget as $task) {
            // 检测用户开启的活动
            if($nowTime >= $task[TemplateTaskTarget::StartTime] && $nowTime
                <= $task[TemplateTaskTarget::EndTime]) {
                // 需要增加任务
                $taskID = $task[TemplateTaskTarget::Id];
                switch ($task[TemplateTaskTarget::TaskType]) {
                    // 累计登录
                    case TemplateTaskTarget::TASK_TYPE_LOGIN_NUM:
                        // 记录登录天数
                        $day = ceil(($nowTime - $task[TemplateTaskTarget::StartTime]) / 86400);
                        $taskExchangeLoginModel->data->loginRecord[] = $day;
                        $taskExchangeLoginModel->data->updateDB($taskID);
                        break;
                    default:
                        $limitTimeModel->data->exchangeRecord[] = $taskID;
                        break;
                }
            }
        }

        $limitTimeModel->data->updateDB();
    }

    public function addExchangeTask(array $exchangeTask)
    {
        $taskBase = new TaskMangerModelBase();
        $taskBase->setPlayerId($this->playerId);
        foreach ($exchangeTask as $taskID) {
            if(is_null($taskBase->getTaskHave($taskID))) {
                $taskBase->createTask($taskID);
            }
        }
    }

    public function delExchangeTask(array $exchangeTask)
    {
        $taskBase = new TaskMangerModelBase();
        $taskBase->setPlayerId($this->playerId);
        foreach ($exchangeTask as $taskID) {
            $taskBase->delTask($taskID);
        }
    }

    public function updateExchangeTask(array $exchangeTask)
    {
        $taskBase = new TaskMangerModelBase();
        $taskBase->setPlayerId($this->playerId);
        foreach ($exchangeTask as $taskID) {
            $taskData = $this->getTitle(TemplateDefine::TYPE_TASK, $taskID);
            if(is_null($taskData)) {
                LogMark::getInstance()->markError(GameErrorCode::DATA_SEARCH_ERROR,
                    '[limitActivityLogic] get task data error! taskID : ' . $taskID);
                continue;
            }
            switch($taskData[TemplateTask::TaskType]) {
                case TemplateTask::TASK_TYPE_LIMIT_TIME_DAY:
                    $taskBase->resetTask($taskID);
                    break;
                case TemplateTask::TASK_TYPE_LIMIT_TIME_WEEK:
                    $loginTime = $this->getAccountAttribute(AccountData::DB_LOGIN_TIME);
                    if(Utils::checkIsCrossWeek($loginTime)) {
                        $taskBase->resetTask($taskID);
                    }
                    break;
                default:
                    break;
            }
        }
    }
}
