<?php

/**
 * 任务数据
 */

namespace Game\Data;

use Framework\DB\Handler\PlayerDBPipe;
use Framework\Log\LogMark;
use Framework\DB\Handler\PlayerDBHandler;
use Framework\Logic\TemplateHelp;
use Game\Constant\ConstTemplate\TemplateTask;
use Game\Constant\DBTableDefine;
use Game\Constant\GameErrorCode;
use Game\Constant\TemplateDefine;

class TaskData
{
    use PlayerDBHandler;
    use TemplateHelp;

    public const TASK_ID = "TaskId";
    public const STATUS = "Status";
    public const SCHEDULE = "Schedule"; // 进度
    public const IS_SYNC = "IsSync";
    public const TRIGGER_DATE = "TriggerDate";  //触发时间,70012任务用

    public const DB_KEY_MAP = array(
        self::TASK_ID => "taskId",
        self::STATUS => "status",
        self::SCHEDULE => "schedule",
        self::IS_SYNC => "isSync",
        self::TRIGGER_DATE => "triggerDate",
    );

    public const STATUS_DOING = 0;  //进行中
    public const STATUS_DONE = 1;   //完成未领取
    public const STATUS_OVER = 2;   //完成已领取

    public ?array $taskConfig = null;
    public bool $isChange = false;  //更新标志,更新后检查是否需要保存

    public int $playerId;
    public int $taskId;
    public int $status;
    public int $schedule;
    public bool $isSync;
    public int $triggerDate = 0;
    public int $carId;
    public int $week;

    private bool $isRegister = false;     //初始创建用户,批量创建任务用
    //设置为初始创建用户
    public function setIsRegister()
    {
        $this->isRegister = true;
    }

    public function __construct(int $playerId)
    {
        $this->playerId = $playerId;
        $this->taskId = 0;
        $this->status = self::STATUS_DOING;
        $this->schedule = 0;
    }

    /**
     * 初始任务
     * @param int $taskId
     * @param array $taskConfig
     * @param array $param 可填参数 CarId Week IsSync
     */
    public function initTask(int $taskId, array $taskConfig, array $param = array())
    {
        $this->taskId = $taskId;
        if (isset($param["CarId"])) {
            $this->setCarId($param["CarId"]);
        }
        if (isset($param["Week"])) {
            $this->setWeek($param["Week"]);
        }
        if (isset($param["IsSync"])) {
            $this->isSync = $param["IsSync"];
        } else {
            $this->isSync = true;
        }
        $this->setConfig($taskConfig);
    }

    //创建
    public function create(): bool
    {
        return $this->save();
    }

    //查询
    public function getTask(int $taskId): bool
    {
        $this->initTaskConfig($taskId);
        switch ($this->getTaskType($taskId))
        {
            case TemplateTask::TASK_TYPE_DAILY:
                return $this->getDailyTask($taskId);
            case TemplateTask::TASK_TYPE_MAIN:
                return $this->getMainTask($taskId);
            case TemplateTask::TASK_TYPE_BRANCH:
                return $this->getBranchTask($taskId);
            case TemplateTask::TASK_TYPE_SEASON_WEEK:
            case TemplateTask::TASK_TYPE_SEASON:
                return $this->getSeasonTask($taskId);
            case TemplateTask::TASK_TYPE_WELFARE_WEEK:
                return $this->getWelfareWeekTask($taskId);
            case TemplateTask::TASK_TYPE_DAILY_CHALLENGE:
                return $this->getDailyChallengeTask($taskId);
            case TemplateTask::TASK_TYPE_FRIEND_INTIMACY:
                return $this->getFriendIntimacyTask($taskId);

            case TemplateTask::TASK_TYPE_LIMIT_TIME:
                return $this->getLimitTimeTask($taskId);
            case TemplateTask::TASK_TYPE_LIMIT_TIME_DAY:
                return $this->getLimitTimeDayTask($taskId);
            case TemplateTask::TASK_TYPE_LIMIT_TIME_WEEK:
                return $this->getLimitTimeWeekTask($taskId);
            default:
                return false;
        }
    }

    public function updateTask(): bool
    {
        $this->initTaskConfig($this->taskId);
        //根据任务类型 分别处理
        switch ($this->getTaskType($this->taskId)) {
            case TemplateTask::TASK_TYPE_DAILY:
                return $this->updateDailyTask();
            case TemplateTask::TASK_TYPE_WELFARE_WEEK:
                return $this->updateWelfareWeekTask();
            case TemplateTask::TASK_TYPE_MAIN:
                return $this->updateMainTask();
            case TemplateTask::TASK_TYPE_BRANCH:
                return $this->updateBranchTask();
            case TemplateTask::TASK_TYPE_SEASON_WEEK:
            case TemplateTask::TASK_TYPE_SEASON:
                return $this->updateSeasonTask();
            case TemplateTask::TASK_TYPE_DAILY_CHALLENGE:
                return $this->updateDailyChallengeTask();
            case TemplateTask::TASK_TYPE_FRIEND_INTIMACY:
                return $this->updateFriendIntimacyTask();

            case TemplateTask::TASK_TYPE_LIMIT_TIME:
                return $this->updateLimitTimeTask();
            case TemplateTask::TASK_TYPE_LIMIT_TIME_DAY:
                return $this->updateLimitTimeDayTask();
            case TemplateTask::TASK_TYPE_LIMIT_TIME_WEEK:
                return $this->updateLimitTimeWeekTask();
            default:
                return false;
        }
    }

    //删除
    public function delTask(int $taskId): bool
    {
        $this->initTaskConfig($taskId);
        //根据任务类型 分别处理
        switch ($this->getTaskType($taskId)) {
            case TemplateTask::TASK_TYPE_DAILY:
                return $this->delDailyTask($taskId);
            case TemplateTask::TASK_TYPE_BRANCH:
                return $this->delBranchTask($taskId);
            case TemplateTask::TASK_TYPE_MAIN:
                return $this->delMainTask($taskId);
            case TemplateTask::TASK_TYPE_SEASON_WEEK:
            case TemplateTask::TASK_TYPE_SEASON:
                return $this->delSeasonTask($taskId);
            case TemplateTask::TASK_TYPE_WELFARE_WEEK:
                return $this->delWelfareWeekTask($taskId);
            case TemplateTask::TASK_TYPE_DAILY_CHALLENGE:
                return $this->delDailyChallengeTask($taskId);
            case TemplateTask::TASK_TYPE_FRIEND_INTIMACY:
                return $this->delFriendIntimacyTask($taskId);

            case TemplateTask::TASK_TYPE_LIMIT_TIME:
                return $this->delLimitTimeTask($taskId);
            case TemplateTask::TASK_TYPE_LIMIT_TIME_DAY:
                return $this->delLimitTimeDayTask($taskId);
            case TemplateTask::TASK_TYPE_LIMIT_TIME_WEEK:
                return $this->delLimitTimeWeekTask($taskId);
            default:
                return false;
        }
    }

    //重置
    public function reset(): bool
    {
        $this->status = self::STATUS_DOING;
        $this->schedule = 0;
        return true;
    }

    private function save(): bool
    {
        $this->initTaskConfig($this->taskId);
        //根据任务类型 分别存储
        switch ($this->getTaskType($this->taskId))
        {
            case TemplateTask::TASK_TYPE_DAILY:
                return $this->saveDailyTask();
            case TemplateTask::TASK_TYPE_WELFARE_WEEK:
                return $this->saveWelfareWeekTask();
            case TemplateTask::TASK_TYPE_BRANCH:
                return $this->saveBranchTask();
            case TemplateTask::TASK_TYPE_MAIN:
                return $this->saveMainTask();
            case TemplateTask::TASK_TYPE_SEASON_WEEK:
            case TemplateTask::TASK_TYPE_SEASON:
                return $this->saveSeasonTask();
            case TemplateTask::TASK_TYPE_DAILY_CHALLENGE:
                return $this->saveDailyChallengeTask();
            case TemplateTask::TASK_TYPE_FRIEND_INTIMACY:
                return $this->saveFriendIntimacyTask();
            case TemplateTask::TASK_TYPE_LIMIT_TIME:
                return $this->saveLimitTimeTask();
            case TemplateTask::TASK_TYPE_LIMIT_TIME_DAY:
                return $this->saveLimitTimeDayTask();
            case TemplateTask::TASK_TYPE_LIMIT_TIME_WEEK:
                return $this->saveLimitTimeWeekTask();
            default:
                return false;
        }
    }

    public function setConfig(array $config)
    {
        $this->taskConfig = $config;
    }

    public function changeSync(bool $sync)
    {
        $this->isSync = $sync;
        $this->save();
    }

    public function setCarId(int $carId)
    {
        $this->carId = $carId;
    }

    public function setWeek(int $week)
    {
        $this->week = $week;
    }

    private function initTaskConfig(int $taskId)
    {
        if (!is_null($this->taskConfig)) {
            return;
        }
        $taskConfig = $this->getTitle(TemplateDefine::TYPE_TASK, $taskId);
        if (is_null($taskConfig)) {
            LogMark::getInstance()->markError(
                GameErrorCode::TASK_ID_NOT_FOUND,
                "[taskData] taskId not found",
                array(
                    "taskId" => $taskId
                )
            );
            return;
        }
        $this->setConfig($taskConfig);
    }

    private function getTaskType(int $taskId): int
    {
        if (!is_null($this->taskConfig)) {
            return $this->taskConfig[TemplateTask::TaskType];
        }
        return (int)substr($taskId, 0, 1);
    }

    private function _getData(): array
    {
        return array(
            self::TASK_ID => $this->taskId,
            self::STATUS => $this->status,
            self::SCHEDULE => $this->schedule,
            self::IS_SYNC => (int)$this->isSync,
            self::TRIGGER_DATE => $this->triggerDate,
        );
    }

    private function formatData(array $result)
    {
        foreach ($result as $key => $value) {
            $val = self::DB_KEY_MAP[$key] ?? $key;
            if ($key == self::IS_SYNC) {
                $this->$val = (bool)(int)$value;
            } else {
                $this->$val = (int)$value;
            }
        }
    }

    private function saveDailyTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_DAILY,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_TASK_DAILY,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

    private function saveWelfareWeekTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_WELFARE_WEEK,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_TASK_WELFARE_WEEK,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

    private function updateDailyTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_DAILY,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_TASK_DAILY,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    private function updateWelfareWeekTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_WELFARE_WEEK,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_TASK_WELFARE_WEEK,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    private function getDailyTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
            DBTableDefine::TABLE_TASK_DAILY,
            $this->playerId,
            $taskId,
            $result
        )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    private function delDailyTask(int $taskId): bool
    {
        return $this->deleteTitle(
            DBTableDefine::TABLE_TASK_DAILY,
            $this->playerId,
            $taskId,
        );
    }

    private function delWelfareWeekTask(int $taskId): bool
    {
        return $this->deleteTitle(
            DBTableDefine::TABLE_TASK_WELFARE_WEEK,
            $this->playerId,
            $taskId,
        );
    }

    //成长任务--(原主线任务)
    private function saveMainTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_MAIN,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_TASK_MAIN,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

    private function updateMainTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_MAIN,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_TASK_MAIN,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    public function getMainTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_MAIN,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    public function getMainTaskList(): array
    {
        $result = array();
        if (!$this->getTitleList(
            DBTableDefine::TABLE_TASK_MAIN,
            $this->playerId,
            $result
        )) {
            return array();
        }
        return $result;
    }

    private function delMainTask(int $taskId): bool
    {
        return $this->deleteTable(
            DBTableDefine::TABLE_TASK_MAIN,
            $this->playerId,
            $taskId,
        );
    }

    //批量创建 会创建出 {x-x-playerId}-TaskBranch 的ZSet格式的表,可忽略
    private function saveBranchTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_BRANCH,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
            );
        }
        //数据
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_BRANCH,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        //索引
        $b2 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_BRANCH,
            $this->playerId,
            $this->carId,
            array(
                $this->taskId => $this->status
            )
        );
        return $b1 && $b2;
    }

    private function updateBranchTask(): bool
    {
        //数据
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_BRANCH,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        //索引
        $b2 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_BRANCH,
            $this->playerId,
            $this->carId,
            array(
                $this->taskId => $this->status
            )
        );
        return $b1 && $b2;
    }

    private function getBranchTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_BRANCH,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        $this->setCarId($this->taskConfig[TemplateTask::GroupType]);
        return true;
    }

    //删除支线任务数据
    private function delBranchTask(int $taskId): bool
    {
        //删除索引
        $ret = 0;
        $b1 = $this->delTermMulti(
            $ret,
            DBTableDefine::TABLE_TASK_BRANCH,
            $this->playerId,
            $this->carId,
            array(
                $taskId
            )
        );
        //删除数据
        $b2 = $this->deleteTitle(
            DBTableDefine::TABLE_TASK_BRANCH,
            $this->playerId,
            $taskId
        );
        return $b1 && $b2;
    }

    private function saveSeasonTask(): bool
    {
        if ($this->isRegister) {
            //数据
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_SEASON,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
            );
        }
        //数据
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_SEASON,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        //索引
        $b2 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_SEASON,
            $this->playerId,
            $this->week,
            array(
                $this->taskId => $this->status
            )
        );
        return $b1 && $b2;
    }

    private function updateSeasonTask(): bool
    {
        //数据
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_SEASON,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        //索引
        $b2 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_SEASON,
            $this->playerId,
            $this->week,
            array(
                $this->taskId => $this->status
            )
        );
        return $b1 && $b2;
    }

    private function getSeasonTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_SEASON,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        $this->setWeek($this->taskConfig[TemplateTask::GroupId]);
        return true;
    }

    private function getWelfareWeekTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_WELFARE_WEEK,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    //删除战令任务数据
    private function delSeasonTask(int $taskId): bool
    {
        //删除索引
        $ret = 0;
        $b1 = $this->delTermMulti(
            $ret,
            DBTableDefine::TABLE_TASK_SEASON,
            $this->playerId,
            $this->week,
            array(
                $taskId
            )
        );
        //删除数据
        $b2 = $this->deleteTitle(
            DBTableDefine::TABLE_TASK_SEASON,
            $this->playerId,
            $taskId
        );
        return $b1 && $b2;
    }

    private function getDailyChallengeTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_DAILY_CHALLENGE,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    private function delDailyChallengeTask(int $taskId): bool
    {
        return $this->deleteTitle(
            DBTableDefine::TABLE_TASK_DAILY_CHALLENGE,
            $this->playerId,
            $taskId,
        );
    }

    private function updateDailyChallengeTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_DAILY_CHALLENGE,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_TASK_DAILY_CHALLENGE,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    private function saveDailyChallengeTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_DAILY_CHALLENGE,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_TASK_DAILY_CHALLENGE,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

    private function getFriendIntimacyTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_FRIEND_INTIMACY_TASK,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    private function getLimitTimeTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_LIMIT_TIME,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    private function getLimitTimeDayTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_LIMIT_TIME_DAY,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    private function getLimitTimeWeekTask(int $taskId): bool
    {
        $result = array();
        if (false == $this->searchTitle(
                DBTableDefine::TABLE_TASK_LIMIT_TIME_WEEK,
                $this->playerId,
                $taskId,
                $result
            )) {
            return false;
        }
        $this->formatData($result);
        return true;
    }


    private function updateFriendIntimacyTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_FRIEND_INTIMACY_TASK,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_FRIEND_INTIMACY_TASK,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    private function updateLimitTimeTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_TASK_LIMIT_TIME,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    private function updateLimitTimeDayTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_DAY,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_DAY,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    private function updateLimitTimeWeekTask(): bool
    {
        $b1 = $this->updateTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_WEEK,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
        );
        $b2 = $this->addRank(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_WEEK,
            $this->playerId,
            $this->taskId,
            $this->status
        );
        return $b1 && $b2;
    }

    private function delFriendIntimacyTask(int $taskId): bool
    {
        return $this->deleteTitle(
            DBTableDefine::TABLE_FRIEND_INTIMACY_TASK,
            $this->playerId,
            $taskId,
        );
    }

    private function delLimitTimeTask(int $taskId): bool
    {
        return $this->deleteTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME,
            $this->playerId,
            $taskId,
        );
    }

    private function delLimitTimeDayTask(int $taskId): bool
    {
        return $this->deleteTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_DAY,
            $this->playerId,
            $taskId,
        );
    }

    private function delLimitTimeWeekTask(int $taskId): bool
    {
        return $this->deleteTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_WEEK,
            $this->playerId,
            $taskId,
        );
    }


    private function saveFriendIntimacyTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_FRIEND_INTIMACY_TASK,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_FRIEND_INTIMACY_TASK,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

    private function saveLimitTimeTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_LIMIT_TIME,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

    private function saveLimitTimeDayTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_LIMIT_TIME_DAY,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_DAY,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

    private function saveLimitTimeWeekTask(): bool
    {
        if ($this->isRegister) {
            return PlayerDBPipe::getInstance()->createTitle(
                DBTableDefine::TABLE_TASK_LIMIT_TIME_WEEK,
                $this->playerId,
                $this->taskId,
                $this->_getData(),
                $this->status
            );
        }
        return $this->createTitle(
            DBTableDefine::TABLE_TASK_LIMIT_TIME_WEEK,
            $this->playerId,
            $this->taskId,
            $this->_getData(),
            $this->status
        );
    }

}
