<?php

/**
 * 玩家邮件
 */

namespace Game\Data;

use Framework\Log\LogMark;
use Framework\DB\Handler\PlayerDBHandler;
use Game\Constant\DBTableDefine;
use Game\Constant\GameErrorCode;

class MailData
{
    use PlayerDBHandler;

    public const MAIL_TPL_ID = "MailTplId";
    public const UID = "Uid";
    public const RECEIVE_TIME = "ReceiveTime";  //接收邮件时间
    public const STATUS = "Status";
    public const EXPIRE_TIME = "ExpireTime";
    public const ENCLOSURE = "Enclosure";   //附件,根据类型处理,默认为空
    public const CONTENT = "Content"; // 邮件内容
    public const FROM_PLAYER_ID = "FromPlayerId";   //发件人Id,系统邮件为0
    public const REPLACE_STRING = "ReplaceString";  //替换内容 json

    public const DB_KEY_MAP = array(
        self::MAIL_TPL_ID => "mailTplId",
        self::UID => "uid",
        self::RECEIVE_TIME => "receiveTime",
        self::STATUS => "status",
        self::EXPIRE_TIME => "expireTime",
        self::ENCLOSURE => "enclosure",
        self::CONTENT => "content",
        self::FROM_PLAYER_ID => "fromPlayerId",
        self::REPLACE_STRING => "replaceString",
    );

    //邮件状态
    public const STATUS_NEW = 0;
    public const STATUS_READ = 1;
    public const STATUS_OVER = 3;   //已读已领附件
    public const STATUS_DEL = 4;    //已删除

    public int $playerId = 0;
    public int $mailTplId;
    public int $uid;
    public int $receiveTime;
    public int $status;
    public int $expireTime;
    public string $enclosure;
    public string $content;
    public int $fromPlayerId;
    public string $replaceString;

    public function __construct(int $playerId, int $uid = 0)
    {
        $this->playerId = $playerId;
        $this->uid = $uid;
        $this->status = self::STATUS_NEW;
        $this->enclosure = "[]";
        $this->content = "";
        $this->replaceString = "[]";
        $this->fromPlayerId = 0;
    }

    public function createMail(): bool
    {
        $data = array();
        foreach (self::DB_KEY_MAP as $k => $v) {
            $data[$k] = $this->$v;
        }
        if(!$this->createTitle(DBTableDefine::TABLE_MAIL,
                               $this->playerId,
                               $this->uid,
                               $data,
                               $this->expireTime
        )) {
            LogMark::getInstance()->markWarn(GameErrorCode::DATA_CREATE_ERROR,
                                              '[MailData] create mail error!',
                                              $data
            );
            return false;
        }
        return true;
    }

    public function getMailData(): bool
    {
        $result = array();
        if(!$this->searchTitle(DBTableDefine::TABLE_MAIL, $this->playerId, $this->uid, $result)) {
            return false;
        }
        if (empty($result)) {
            return false;
        }
        $this->formatData($result);
        return true;
    }

    public function saveDB(): bool
    {
        $data = array();
        foreach (self::DB_KEY_MAP as $k => $v) {
            $data[$k] = $this->$v;
        }
        if(!$this->updateTitle(DBTableDefine::TABLE_MAIL, $this->playerId, $this->uid, $data)) {
            LogMark::getInstance()->markWarn(
                GameErrorCode::DATA_UPDATE_ERROR,
              '[MailData] update mail error!',
                      $data);
            return false;
        }
        return true;
    }

    public function delMail(): bool
    {
        if(!$this->deleteTitle(DBTableDefine::TABLE_MAIL, $this->playerId, $this->uid)) {
            LogMark::getInstance()->markWarn(
                GameErrorCode::DATA_DELETE_ERROR,
                '[MailData] delete mail error!',
                array(
                    "playerId" => $this->playerId,
                    "Uid" => $this->uid
                )
            );
            return false;
        }
        return true;
    }


    private function formatData($result)
    {
        foreach ($result as $key => $value) {
            $val = self::DB_KEY_MAP[$key] ?? $key;
            if (is_numeric($value)) {
                $this->$val = intval($value);
            } else {
                $this->$val = $value;
            }
        }
    }
}
