<?php

/**
 * ItemData.php
 * 道具数据
 * Created On 2022/1/25
 */

namespace Game\Data;

use Framework\DB\Handler\PlayerDBHandler;
use Framework\DB\Handler\PlayerDBPipe;
use Framework\Logic\TimerLogic;
use Game\Constant\DBTableDefine;
use Framework\Log\LogMark;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;

class ItemData
{
    use PlayerDBHandler;
    use TimerLogic;

    public const ITEM_ID = "ItemId";
    public const EXPIRATION_TIME = "ExpirationTime";
    public const NUM = "Num";
    public const ID = "Id";

    public const DB_KEY_MAP = array(
        self::ITEM_ID => "tplID",
        self::EXPIRATION_TIME => "expire",
        self::NUM => "num",
        self::ID => "uid",
    );

    public int $playerID = 0;
    public int $uid;
    public int $tplID;
    public int $num;
    public int $expire;

    public function __construct(int $playerID, int $uid = 0, int $tplID = 0, int $num = 0, int $expire = 0) {
        $this->playerID = $playerID;
        $this->uid = $uid;
        $this->tplID = $tplID;
        $this->num = $num;
        $this->expire = $expire;
    }

    public function searchItem():bool {
        $itemArr = [];
        if(!$this->searchTitle(DBTableDefine::TABLE_ITEM, $this->playerID, $this->uid, $itemArr)) {
            return false;
        }
        if (empty($itemArr)) {
            return false;
        }
        $this->setExpire(intval($itemArr['ExpirationTime']));
        $this->setTplId(intval($itemArr['ItemId']));
        $this->setNum(intval($itemArr['Num']));
        return true;
    }

    public function createItem(): bool
    {
        $data = array();
        foreach (self::DB_KEY_MAP as $k => $v) {
            $data[$k] = $this->$v;
        }

        // 限时道具则增加计时器
        if ($this->expire != 0) {
            $this->addTimer(ModelTypeDefine::ITEM, $this->uid, $this->expire);
        }
        if(!$this->createTitle(DBTableDefine::TABLE_ITEM, $this->playerID, $this->uid, $data, $this->tplID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[data] item create error!', $data
            );
            return false;
        }
        return true;
    }

    //批量创建道具
    public function registerDB(): bool
    {
        $data = array();
        foreach (self::DB_KEY_MAP as $k => $v) {
            $data[$k] = $this->$v;
        }
        return PlayerDBPipe::getInstance()->createTitle(
            DBTableDefine::TABLE_ITEM,
            $this->playerID,
            $this->uid,
            $data,
            $this->tplID
        );
    }

    public function saveDB(): bool
    {
        $data = array();
        foreach (self::DB_KEY_MAP as $k => $v) {
            $data[$k] = $this->$v;
        }

        // 更新计时器时间
        if ($this->expire != 0) {
            $this->addTimer(ModelTypeDefine::ITEM, $this->uid, $this->expire);
        }
        if(!$this->updateTitle(DBTableDefine::TABLE_ITEM, $this->playerID, $this->uid, $data)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[data] item update error!', $data
            );
            return false;
        }
        return true;
    }

    public function deleteItem(): bool {

        // 删除过期数据
        if ($this->expire != 0) {
            $this->delTimer(ModelTypeDefine::ITEM, $this->uid);
        }

        if(!$this->deleteTitle(DBTableDefine::TABLE_ITEM, $this->playerID, $this->uid)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_DELETE_ERROR,
                '[data] item delete error!',
                ['playerID' => $this->playerID, 'uid' => $this->uid]
            );
            return false;
        }
        return true;
    }

    public function setTplId(int $tplID) {
        $this->tplID = $tplID;
    }

    public function setExpire(int $expire) {
        $this->expire = $expire;
    }

    public function setNum(int $Num) {
        $this->num = $Num;
    }
}
