<?php

/**
 * CarData.php
 * 车辆数据
 */

namespace Game\Data\Car;

use Framework\DB\Handler\PlayerDBHandler;
use Framework\Logic\TimerLogic;
use Game\Constant\DBTableDefine;
use Framework\Log\LogMark;
use Game\Constant\GameErrorCode;
use Framework\Logic\TemplateHelp;
use Game\Constant\ModelTypeDefine;

class CarData
{
    use PlayerDBHandler;
    use TimerLogic;
    use TemplateHelp;

    //
    public const UID    = "Uid";
    public const CAR_ID = "CarID";
    public const LEVEL  = "Level";
    public const ACTIVE_SKILL  = "ActiveSkill";
    public const PASSIVE_SKILL  = "PassiveSkill";
    public const PROFICIENCY = "Proficiency";      //熟练度
    public const PROFICIENCY_LV = "ProficiencyLv";      //熟练度等级
    public const PROFICIENCY_REWARD = "ProficiencyReward";      //熟练度奖励进度[奖励id,..]
    public const CREATE_TIME = "CreateTime";    //获得时间
    public const EXPIRE = "Expire";
    public const STATE  = "State";
    public const PERFORMANCE  = "Performance";

    public const DB_KEY_MAP = array(
        self::UID => "uid",
        self::CAR_ID => "carID",
        self::LEVEL => "level",
        self::ACTIVE_SKILL => "activeSkill",
        self::PASSIVE_SKILL => "passiveSkill",
        self::PROFICIENCY => "proficiency",
        self::EXPIRE => "expire",
        self::STATE => "state",
        self::PERFORMANCE => "performance",
        self::PROFICIENCY_LV => "proficiencyLv",
        self::PROFICIENCY_REWARD => "proficiencyReward",
        self::CREATE_TIME => "createTime",
    );

    public int $playerID = 0;
    public int $uid;
    public int $carID;
    public int $level;
    public int $activeSkill;
    public int $proficiency;
    public string $passiveSkill;
    public int $expire;
    public int $state;
    public int $performance;
    public int $proficiencyLv;
    public string $proficiencyReward;
    public int $createTime;

    //初始熟练度为1级
    public function __construct(
        int $playerID, int $uid = 0, int $carID = 0,
        int $level = 1, int $expire = 0, int $state = 0,
        int $activeSkill = 0, string $passiveSkill = '',
        int $proficiency = 0, int $performance = 0
    ) {
        $this->playerID = $playerID;
        $this->uid = $uid;
        $this->carID = $carID;
        $this->level = $level;
        $this->expire = $expire;
        $this->state = $state;
        $this->activeSkill = $activeSkill;
        $this->passiveSkill = $passiveSkill;
        $this->proficiency = $proficiency;
        $this->performance = $performance;
        $this->proficiencyLv = 1;
    }

    public function searchCar():bool {
        $car = [];
        if(!$this->searchTitle(DBTableDefine::TABLE_CAR, $this->playerID, $this->uid, $car)) {
            return false;
        }
        if (empty($car)) {
            return false;
        }
        $this->setCarID((int)$car[self::CAR_ID]);
        $this->setLevel((int)$car[self::LEVEL]);
        $this->setActiveSkill((int)$car[self::ACTIVE_SKILL]);
        $this->setPassiveSkill($car[self::PASSIVE_SKILL]);
        $this->setExpire((int)$car[self::EXPIRE]);
        $this->setState((int)$car[self::STATE]);
        $this->setProficiency((int)$car[self::PROFICIENCY]);
        $this->setPerformance($car[self::PERFORMANCE] ?? 0);
        $this->setProficiencyLv($car[self::PROFICIENCY_LV] ?? 0);
        $this->setProficiencyReward($car[self::PROFICIENCY_REWARD] ?? "[]");
        $this->setCreateTime($car[self::CREATE_TIME] ?? 0);
        return true;
    }

    public function createCar(): bool
    {
        $data = array();
        foreach (self::DB_KEY_MAP as $k => $v) {
            $data[$k] = $this->$v;
        }

        // 限时道具则增加计时器
        if ($this->expire != 0) {
            $this->addTimer(ModelTypeDefine::CAR, $this->uid, $this->expire);
        }
        if(!$this->createTitle(DBTableDefine::TABLE_CAR, $this->playerID, $this->uid, $data, $this->carID)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_CREATE_ERROR,
                '[data] car create error!', $data
            );
            return false;
        }
        return true;
    }

    public function saveDB(): bool
    {
        $data = array();
        foreach (self::DB_KEY_MAP as $k => $v) {
            $data[$k] = $this->$v;
        }
        if(!$this->updateTitle(DBTableDefine::TABLE_CAR, $this->playerID, $this->uid, $data)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_UPDATE_ERROR,
                '[data] car update error!', $data
            );
            return false;
        }
        return true;
    }

    public function deleteCar(): bool {
        if(!$this->deleteTitle(DBTableDefine::TABLE_CAR, $this->playerID, $this->uid)) {
            LogMark::getInstance()->markError(
                GameErrorCode::DATA_DELETE_ERROR,
                '[data] car delete error!',
                ['playerID' => $this->playerID, 'uid' => $this->uid]
            );
            return false;
        }
        return true;
    }

    public function setCarID(int $carID) {
        $this->carID = $carID;
    }

    public function setExpire(int $expire) {
        $this->expire = $expire;
    }

    public function setLevel(int $level) {
        $this->level = $level;
    }

    public function setActiveSkill(int $activeSkill) {
        $this->activeSkill = $activeSkill;
    }

    public function setPerformance(int $performance) {
        $this->performance = $performance;
    }

    public function setPassiveSkill(string $passiveSkill) {
        $this->passiveSkill = $passiveSkill;
    }

    public function setState(int $state) {
        $this->state = $state;
    }

    public function setProficiency(int $proficiency) {
        $this->proficiency = $proficiency;
    }

    public function setProficiencyLv(int $lv)
    {
        $this->proficiencyLv = $lv;
    }

    public function setProficiencyReward(string $str)
    {
        $this->proficiencyReward = $str;
    }

    public function setCreateTime(int $createTime)
    {
        $this->createTime = $createTime;
    }

    public function getIndexId(int $carID, int $level): int {
        return $carID * 100 + $level;
    }

    public function getPassiveSkill(array $carInfo): array {
        $passiveSkill = $carInfo[self::PASSIVE_SKILL];
        $passiveSkillArr = explode(';', $passiveSkill);
        return array_values($passiveSkillArr);
    }
}
