<?php

/**
 * 赛季周
 */

namespace Framework\Model;

use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\Logic\CommonDataLogic;
use Framework\Logic\CommonTimerLogic;
use Framework\Logic\TemplateHelp;
use Game\Constant\ConstTemplate\TemplateMatchTime;
use Game\Constant\GameConstantDefine;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Logic\CommonConstLogic;

class CommonSeasonWeekModel
{
    use TemplateHelp;
    use CommonTimerLogic;
    use CommonDataLogic;
    use CommonConstLogic;

    //设置赛季所在周
    public function setSeasonWeek(int $week)
    {
        LogMark::getInstance()->markInfo(
            "[CommonSeasonWeekModel] setSeasonWeek",
            array(
                "week" => $week
            )
        );
        if ($week == 0) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_SEASON_WEEK_ERROR,
                "[CommonSeasonWeekModel] setSeasonWeek error,week is 0",
                array(
                    "week" => $week
                )
            );
        }
        //添加全局数据
        $this->addCommonData($this->getPlayerModel(), $week);

        //添加过期时间
        $this->addCommonTimer(ModelTypeDefine::COMMON_SEASON_WEEK, 0, $this->getNextMondayTime());

        $this->setSeasonWeekToCommonConst($week);
    }

    //删除过期 更新周
    public function commonTimerExpire(array $expireData): bool
    {
        if(empty($expireData)) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_TIMER_EXPIRE_DATA_ERROR,
                '[CommonSeasonWeekModel] timerExpire array empty!');
            return false;
        }
        if (count($expireData) != 1) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_TIMER_EXPIRE_DATA_ERROR,
                '[CommonSeasonWeekModel] timerExpire array error!',
                $expireData
            );
            return false;
        }
        $week = $this->getSeasonWeek();
        $this->setSeasonWeek($week);
        return true;
    }

    //设置赛季所在周
    private function setSeasonWeekToCommonConst(int $week)
    {
        $data = array(
            GameConstantDefine::COMMON_KEY_SEASON_WEEK => $week
        );
        $this->saveCommonConstData($data);
    }

    private function getSeasonWeek(): int
    {
        $result = $this->getCommonConstData();
        //计算当前时间为赛季第几周
        $seasonId = (int)$result[GameConstantDefine::COMMON_KEY_SEASON_ID];
        return $this->calculateSeasonWeek($seasonId);
    }

    //计算当前时间为赛季第几周
    public function calculateSeasonWeek(int $seasonId): int
    {
        $time = Utils::getServerTimestamp();
        $seasonDataConfig = $this->getSeasonConfig($seasonId);
        if (empty($seasonDataConfig)) {
            return 0;
        }
        //检查是否在赛季时间内
        if (!Utils::checkIsCrossDay($seasonDataConfig[TemplateMatchTime::StartTime])) {
            //当前时间和开始时间为同一天
            return 1;
        }
        if ($time < $seasonDataConfig[TemplateMatchTime::StartTime]) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_SEASON_TIME_ERROR,
                "[CommonSeasonWeekModel] time error,season not start",
                array(
                    "time" => $time,
                    "seasonStartTime" => $seasonDataConfig[TemplateMatchTime::StartTime],
                    "seasonId" => $seasonId
                )
            );
            return 0;
        }
        if ($time > $seasonDataConfig[TemplateMatchTime::EndTime]) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_SEASON_TIME_ERROR,
                "[CommonSeasonWeekModel] time error,season is over",
                array(
                    "time" => $time,
                    "seasonStartTime" => $seasonDataConfig[TemplateMatchTime::StartTime],
                    "seasonId" => $seasonId
                )
            );
            return 0;
        }
        $seasonStartTime = $seasonDataConfig[TemplateMatchTime::StartTime];
        $seasonStartWeek = date("W", $seasonStartTime);
        $nowWeek = date("W", $time);
        if ($nowWeek >= $seasonStartWeek) {
            $week = $nowWeek - $seasonStartWeek + 1;
        } else {
            //跨年了--ISO 8601
            $year = (int)date("Y") - 1;
            $lastYearWeek = date("W", strtotime((string)$year ."-12-28"));
            $week = $lastYearWeek - $seasonStartWeek + $nowWeek + 1;
        }
        return $week;
    }

    //获取赛季配置
    private function getSeasonConfig(int $seasonId): array
    {
        $config = $this->getTitle(TemplateDefine::TYPE_MATCH_TIME, $seasonId);
        if (is_null($config)) {
            LogMark::getInstance()->markError(
                GameErrorCode::SEASON_ID_NOT_FOUND,
                "[CommonSeasonWeekModel] season id not found",
                array(
                    "seasonId" => $seasonId
                )
            );
            return array();
        }
        return $config;
    }

    //获取下周一凌晨时间
    private function getNextMondayTime(): int
    {
        $weekDay = date("N");   //1-7 周1-周日
        $day = 8 - $weekDay;
        return strtotime(date("Y-m-d")."+{$day} days") + 1;
    }

    //玩家战令赛季model
    private function getPlayerModel(): int
    {
        return ModelTypeDefine::TASK_SEASON;
    }
}

