<?php

/**
 * 系统邮件
 */

namespace Framework\Model;

use Framework\Lib\Utils;
use Framework\Log\LogMark;
use Framework\DB\Handler\CommonDBHandler;
use Framework\Logic\CommonDataLogic;
use Framework\Logic\CommonTimerLogic;
use Framework\Logic\TemplateHelp;
use Game\Constant\ConstTemplate\TemplateConst;
use Game\Constant\DBTableDefine;
use Game\Constant\GameErrorCode;
use Game\Constant\ModelTypeDefine;
use Game\Constant\TemplateDefine;
use Game\Data\MailData;

class CommonMailModel
{
    use CommonDBHandler;
    use TemplateHelp;
    use CommonTimerLogic;
    use CommonDataLogic;

    private const TITLE_ID = 1;
    private const INDEX_KEY = 0;  //保存邮件最新索引的key

    public function createCommonMail(int $tplId, int $receiveTime): bool
    {
        if ($receiveTime == 0) {
            $receiveTime = Utils::getServerTimestamp();
        }
        //过期时间
        $configExpireTime = $this->getTerm(TemplateDefine::TYPE_CONST,
                                           TemplateConst::Const_15_MailExpireTime,
                                           TemplateConst::ConstNum);
        $expireTime = $receiveTime + $configExpireTime;
        $data = array(
            MailData::MAIL_TPL_ID => $tplId,
            MailData::RECEIVE_TIME => $receiveTime,
        );
        $jsonStr = json_encode($data);
        //获取当前索引
        $nowIndex = $this->getCommonMailNowIndex();
        //更新索引邮件
        if (false == $this->autoAddCommonTerm(
            $nowIndex,
            $this->getDBTable(),
            self::TITLE_ID,
            self::INDEX_KEY,
            1
        )) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_MAIL_UPDATE_INDEX_ERROR,
                "[CommonMailModel] update new index error",
                array(
                    'tplId' => $tplId,
                    'nowIndex' => $nowIndex
                )
            );
            return false;
        }
        $newIndex = $nowIndex;
        //保存邮件
        if (false == $this->createTerm(
            $this->getDBTable(),
            self::TITLE_ID,
            $newIndex,
            $jsonStr
        )) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_MAIL_CREATE_NEW_MAIL_ERROR,
                "[CommonMailModel] create new mail fail",
                array(
                    'tplId' => $tplId,
                    'data' => $data
                )
            );
            return false;
        }
        //添加系统邮件过期
        $this->addCommonTimer($this->getModelType(), $newIndex, $expireTime);
        //更新系统模块索引 更新的应该是玩家的model
        $this->addCommonData(ModelTypeDefine::MAIL, $newIndex);
        return true;
    }

    public function getCommonMailByIndex(array $indexArr): array
    {
        $result = array();
        if (false == $this->searchTermMultiCommon(
            $result,
            $this->getDBTable(),
            self::TITLE_ID,
            $indexArr
        )) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_MAIL_SEARCH_ERROR,
                "[CommonMailModel] getCommonMailIndex fail",
                $indexArr
            );
        }
        return $result;
    }

    public function delCommonMail(array $indexArr): bool
    {
        $tmp = 0;
        if (false == $this->delTermMultiCommon(
            $tmp,
            $this->getDBTable(),
            self::TITLE_ID,
            $indexArr
        )) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_MAIL_DEL_ERROR,
                "[CommonMailModel] delCommonMail fail",
                $indexArr
            );
            return false;
        }
        return true;
    }

    //删除过期
    public function commonTimerExpire(array $expireData): bool
    {
        if(empty($expireData)) {
            LogMark::getInstance()->markError(
                GameErrorCode::COMMON_TIMER_EXPIRE_DATA_ERROR,
                '[CommonMailModel] timerExpire array empty!');
            return false;
        }
        $indexArr = array_keys($expireData);
        return $this->delCommonMail($indexArr);
    }

    private function getCommonMailNowIndex(): int
    {
        $ret = "";
        if (false == $this->searchCommonTerm(
            $ret,
            $this->getDBTable(),
            self::TITLE_ID,
            self::INDEX_KEY
            )
        ) {
            return 0;
        }
        return is_numeric($ret) ?: 0;
    }

    private function getDBTable(): int
    {
        return DBTableDefine::TABLE_COMMON_MAIL;
    }

    private function getModelType(): int
    {
        return ModelTypeDefine::COMMON_MAIL;
    }
}
