<?php

/**
 * TemplateHelp.php
 * 文件描述
 * Created On 2022/1/18
 * @author yuanb yuanbo0x@gmail.com
 */

namespace Framework\Logic;

use Framework\Log\LogMark;
use Game\Constant\ConstTemplate\TemplateUidIndex;
use Game\Constant\GameErrorCode;
use Game\Constant\TemplateDefine;
use Library\MemoryCache;
use Game\Config\GameConfig;

trait TemplateHelp
{
    public function getTitleIdList($templateType): ?array
    {
        $cacheId = $this->makeCacheId($templateType);
        if (false == MemoryCache::GetInstance()->InitCache($cacheId)) {
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getTitleIdList InitCache fail",
                array(
                    "templateType" => $templateType,
                    "cacheId" => $cacheId,
                )
            );
            return null;
        }
        $result = MemoryCache::GetInstance()->GetCache(0, $cacheId);
        if ($result === null) {
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getTitleIdList GetCache is null",
                array(
                    "templateType" => $templateType,
                    "cacheId" => $cacheId,
                    "errMsg" => MemoryCache::getInstance()->getLastError()
                )
            );
            return null;
        }
        return $result;
    }

    public function getTable($templateType): ?array
    {
        $idList = $this->getTitleIdList($templateType);
        if ($idList == null) {
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getTable is null",
                array(
                    "templateType" => $templateType,
                )
            );
            return null;
        }
        $ret = array();
        foreach ($idList as $id) {
            $tpl = $this->getTitle($templateType, $id);
            if ($tpl == null) {
                continue;
            }
            $ret[$id] = $tpl;
        }
        return $ret;
    }

    public function getTitle($templateType, $titleId): ?array
    {
        $cacheId = $this->makeCacheId($templateType);
        if (false == MemoryCache::GetInstance()->InitCache($cacheId)) {
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getTitle InitCache fail",
                array(
                    "templateType" => $templateType,
                    "cacheId" => $cacheId,
                    "titleId" => $titleId,
                )
            );
            return null;
        }
        $result = MemoryCache::GetInstance()->GetCache($titleId, $cacheId);
        if ($result === null) {
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getTitle GetCache is null",
                array(
                    "templateType" => $templateType,
                    "cacheId" => $cacheId,
                    "titleId" => $titleId,
                    "errMsg" => MemoryCache::getInstance()->getLastError()
                )
            );
            return null;
        }
        return $result;
    }

    public function getTerm($templateType, $titleId, $term)
    {
        $ret = $this->getTitle($templateType, $titleId);
        if (!is_array($ret)) {
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getTerm is null",
                array(
                    "templateType" => $templateType,
                )
            );
            return null;
        }
        if (!array_key_exists($term, $ret)) {
            return null;
        }
        return $ret[$term];
    }

    public function getUidIndex(): int
    {
        $cacheId = $this->makeCacheId(TemplateDefine::TYPE_UID_INDEX);
        if (false == MemoryCache::GetInstance()->InitCache($cacheId)) {
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getUidIndex InitCache fail",
                array(
                    "templateType" => TemplateDefine::TYPE_UID_INDEX,
                    "cacheId" => $cacheId,
                    "errMsg" => MemoryCache::getInstance()->getLastError()
                )
            );
            return random_int(1, 32768);
        }
        //尝试10次,如果错误就返回随机数
        $loop = 10;
        while ($loop > 0) {
            $index = MemoryCache::GetInstance()->getUidIndexWithLock(
                TemplateUidIndex::DEFAULT_ID, $cacheId);
            if (!is_null($index)) {
                LogMark::getInstance()->markDebug(
                    "[TemplateHelp] getUidIndex",
                    array(
                        "index" => $index
                    )
                );
                return $index;
            }
            LogMark::getInstance()->markError(
                GameErrorCode::SHM_CACHE_ERROR,
                "[TemplateHelp] getUidIndex is null",
                array(
                    "templateType" => TemplateDefine::TYPE_UID_INDEX,
                    "cacheId" => $cacheId,
                    "errMsg" => MemoryCache::getInstance()->getLastError()
                )
            );
            $loop--;
            usleep(5);
        }
        return random_int(1, 32768);
    }

    private function makeCacheId($type): int
    {
        return GameConfig::getInstance()->DATA_VERSION()  * 1000 + $type;
    }
}
