<?php

namespace Framework\DB\Handler;

use Framework\DB\DBManager;
use Framework\Log\LogMark;
use Framework\Lib\Utils;
use Game\Constant\DBTableDefine;
use Game\Data\RoomCustomizeData;

trait PlayerDBHandler
{
    public function createTitle($table, $playerId, $titleId, $data, $score = 0): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array($dbKey, $titleId, $score);
        foreach ($data as $k => $v) {
            array_push($param, $k);
            array_push($param, $v);
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_PLAYER_CREATE_TITLE,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function searchTable($table, $playerId, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array($dbKey);
        $result = array();
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_PLAYER_SEARCH_TABLE,
                $table,
                $param,
                $result
            )
        ) {
            return false;
        }
        return true;
    }

    public function searchTitleMulti($table, $playerId, $titleIdList, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = $titleIdList;
        array_unshift($param, $dbKey);
        $result = array();
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_PLAYER_SEARCH_TITLE_MULTI,
                $table,
                $param,
                $result
            )
        ) {
            return false;
        }
        return true;
    }

    // 创建房间
    public function createRoomDB($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            $param[] = $v;
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_CREATE_ROOM,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    public function joinRoom($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            $param[] = $v;
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_JOIN_ROOM,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    public function exitRoomDB($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            $param[] = $v;
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_EXIT_ROOM,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    // 创建房间
    public function createCustomizeRoom($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $dbKey .= '-' . $data[RoomCustomizeData::DB_KEY_MAP[RoomCustomizeData::ROOM_ID]];
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            $param[] = $v;
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_CREATE_ROOM_CUSTOMIZE,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    public function joinCustomizeRoom($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $dbKey .= '-' . $data[RoomCustomizeData::DB_KEY_MAP[RoomCustomizeData::ROOM_ID]];
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            $param[] = $v;
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_JOIN_ROOM_CUSTOMIZE,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    public function exitCustomizeRoom($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $dbKey .= '-' . $data[RoomCustomizeData::DB_KEY_MAP[RoomCustomizeData::ROOM_ID]];
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            $param[] = $v;
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_EXIT_ROOM_CUSTOMIZE,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    // ins queue
    public function insQueueRPush($table, $sendServer, $data): bool
    {
        $dbKey = Utils::makeDBKey($table, 0, $sendServer);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_RPUSH,
            $dbKey, $data
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }


    // 创建俱乐部
    public function createClubDB($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            array_push($param, $v);
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_CREATE_CLUB,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    public function joinClubDB($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);

        $param = array($dbKey);
        foreach ($data as $k => $v) {
            array_push($param, $v);
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_JOIN_CLUB,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    public function exitClubDB($table, $playerId, $titleId, $data): ?int
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array($dbKey);
        foreach ($data as $k => $v) {
            array_push($param, $v);
        }
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_EXIT_CLUB,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return $ret;
    }

    public function deleteTitle($table, $playerId, $titleId): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array($dbKey);
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_PLAYER_DELETE_TITLE,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function deleteServerTitle($table, $playerId, $titleId, $serverID): bool
    {
        $dbKey = Utils::makeServerDBKey($table, $playerId, $titleId, $serverID);
        $param = array($dbKey);
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_PLAYER_DELETE_TITLE,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function updateTitle($table, $playerId, $titleId, $data): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_HMSET,
            $dbKey,
            $data
        );
        $ret = "";
        return false != DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            );
    }

    public function updateCRoomTitle($table, $playerId, $titleId, $data): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $dbKey .= '-' . $data[RoomCustomizeData::ROOM_ID];
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_HMSET,
            $dbKey,
            $data
        );
        $ret = "";
        return false != DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            );
    }

    public function updateServerTitle($table, $playerId, $titleId, $serverID, $data): bool
    {
        $dbKey = Utils::makeServerDBKey($table, $playerId, $titleId, $serverID);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_HMSET,
            $dbKey,
            $data
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function searchTitle($table, $playerId, $titleId, &$result): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_HGETALL,
            $dbKey
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function searchCRoomTitle($table, $playerId, $titleId, $roomID, &$result): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $dbKey .= '-' . $roomID;
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_HGETALL,
            $dbKey
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function searchServerTitle($table, $playerId, $titleId, $serverID, &$result): bool
    {
        $dbKey = Utils::makeServerDBKey($table, $playerId, $titleId, $serverID);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_HGETALL,
            $dbKey
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function getTitleList($table, $playerId, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZRANGE,
            $dbKey, 0, -1, true
        );
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    // zAdd player
    public function addPlayerRank($table, $playerId, $titleID, $key, $score): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleID);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZADD,
            $dbKey, $key, $score);
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    // ZScore
    public function searchRankScoreByKey($table, $playerID, $titleId, $key, &$ret): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerID, $titleId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZSCORE,
            $dbKey, $key
        );
        $ret = '';
        return false != DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            );
    }

    public function searchTerm(&$ret, $tableType, $playerId, $titleId, $key): bool
    {
        $dbKey = Utils::makeDBKey($tableType, $playerId, $titleId);
        $param = array(DBTableDefine::DB_LOGIC_CMD_HGET, $dbKey, $key);
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $tableType,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function searchRankByIndex($table, $playerId, $start, $stop, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_Z_REV_RANGE,
            $dbKey, $start, $stop, true);
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function searchRevRankByIndex($table, $playerId, $start, $stop, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_Z_REV_RANGE_BY_SCORE,
            $dbKey, $stop, $start, true);
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function searchRankNum($table, $playerId, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZCARD,
            $dbKey);
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = $ret;
        return true;
    }

    public function addRank($table, $playerId, $key, $score): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZADD,
            $dbKey, $key, $score);
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function delRank($table, $playerId, $key): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZREM,
            $dbKey, $key);
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function searchRank($table, $playerId, $start, $stop, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZRANGE_BY_SCORE,
            $dbKey, $start, $stop, true
        );
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function addQueue($table, $playerId, $term): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_RPUSH,
            $dbKey, $term
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function searchQueue($table, $playerId, $start, $stop, &$result): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_LRANGE,
            $dbKey, $start, $stop
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function deleteQueue($table, $playerId, $start, $stop): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_LTRIM,
            $dbKey, $start, $stop
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function RemQueue($table, $playerId, $key): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_LTRIM,
            $dbKey, $key
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_CMD_LREM,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function deleteQueueTitle($table, $playerId): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_DEL,
            $dbKey
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function deleteTable($table, $playerId, $titleId): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_DEL,
            $dbKey
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function popQueue($table, $playerId, $titleId, &$result): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_LPOP,
            $dbKey
        );
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function searchTermMulti(&$ret, $table, $playerId, $titleId, array $keyArr): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(DBTableDefine::DB_LOGIC_CMD_HMGET, $dbKey, $keyArr);
        if (false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )) {
            return false;
        }
        $ret = json_decode($ret, true);
        return true;
    }

    public function delTermMulti(&$ret, $tableType, $playerId, $titleId, array $keyArr): bool
    {
        $dbKey = Utils::makeDBKey($tableType, $playerId, $titleId);

        $param = array(DBTableDefine::DB_LOGIC_CMD_HDel, $dbKey, $keyArr);
        return false != DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $tableType,
                $param,
                $ret
            );
    }

    public function delHDelTermMulti($tableType, $playerId, $titleId, array $keyArr): bool
    {
        $dbKey = Utils::makeFullServerKey($tableType, $playerId, $titleId);
        $param = array(DBTableDefine::DB_LOGIC_CMD_HDel, $dbKey, $keyArr);
        return false != DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $tableType,
                $param,
                $ret
            );
    }

    public function searchTermLen($table, $playerId, $titleId, &$result): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_HLEN,
            $dbKey
        );
        $ret = "";
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function autoAddTerm($table, $playerId, $titleId, $key, $val, &$result): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(DBTableDefine::DB_LOGIC_CMD_HINCRBY, $dbKey, $key, $val);
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $result = json_decode($ret, true);
        return true;
    }

    public function autoAddTermByRoomID(&$ret, $tableType, $titleId, $key, $val): bool
    {
        $dbKey = Utils::makeDBKey($tableType, $titleId, 0);
        $param = array(DBTableDefine::DB_LOGIC_CMD_HINCRBY, $dbKey, $key, $val);
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $tableType,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function autoAddTermByClubID(&$ret, $tableType, $key, $val): bool
    {
        $dbKey = Utils::makeDBKey($tableType, 0, 0);
        $param = array(DBTableDefine::DB_LOGIC_CMD_HINCRBY, $dbKey, $key, $val);
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $tableType,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function checkTermExists($table, $playerId, $titleId, $key, &$ret): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerId, $titleId);
        $param = array(DBTableDefine::DB_LOGIC_CMD_HEXISTS, $dbKey, $key);
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        $ret = json_decode($ret, true);
        return true;
    }

    public function setMember($table, $playerID, $titleId, $key): bool
    {
        $dbKey = Utils::makeDBKey($table, $playerID, $titleId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_SET,
            $dbKey, $key);
        $ret = '';
        if (
            false == DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            )
        ) {
            return false;
        }
        return true;
    }

    public function searchRankScoreInDBIndex($table, $playerId, $key, &$ret): bool
    {
        $dbKey = Utils::makeDBIndex($table, $playerId);
        $param = array(
            DBTableDefine::DB_LOGIC_CMD_ZSCORE,
            $dbKey, $key
        );
        $ret = '';
        return false != DBManager::getInstance()->executeRedis(
                DBTableDefine::DB_LOGIC_COMMON_CMD,
                $table,
                $param,
                $ret
            );
    }
}
